<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\Course;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\View\View;

class AdminCourseController extends Controller
{
    public function index(): View
    {
        $courses = Course::with('category')->latest()->paginate(15);
        return view('admin.courses.index', compact('courses'));
    }

    public function create(): View
    {
        $categories = Category::where('is_active', true)->orderBy('sort_order')->get();
        return view('admin.courses.create', compact('categories'));
    }

    public function store(Request $request): RedirectResponse
    {
        $data = $request->validate([
            'category_id' => 'required|exists:categories,id',
            'title' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'syllabus' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'type' => 'required|in:free,paid',
            'level' => 'in:beginner,intermediate,advanced,all',
            'exam_type' => 'nullable|string|max:100',
            'total_lessons' => 'integer|min:0',
            'total_duration' => 'integer|min:0',
            'is_featured' => 'boolean',
            'is_active' => 'boolean',
        ]);
        $data['slug'] = $data['slug'] ?? Str::slug($data['title']);
        $data['is_featured'] = $request->boolean('is_featured', false);
        $data['is_active'] = $request->boolean('is_active', true);
        Course::create($data);
        return redirect()->route('admin.courses.index')->with('success', 'Course created.');
    }

    public function show(Course $course): View
    {
        $course->load(['category', 'lessons']);
        return view('admin.courses.show', compact('course'));
    }

    public function edit(Course $course): View
    {
        $categories = Category::where('is_active', true)->orderBy('sort_order')->get();
        return view('admin.courses.edit', compact('course', 'categories'));
    }

    public function update(Request $request, Course $course): RedirectResponse
    {
        $data = $request->validate([
            'category_id' => 'required|exists:categories,id',
            'title' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'syllabus' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'type' => 'required|in:free,paid',
            'level' => 'in:beginner,intermediate,advanced,all',
            'exam_type' => 'nullable|string|max:100',
            'total_lessons' => 'integer|min:0',
            'total_duration' => 'integer|min:0',
            'is_featured' => 'boolean',
            'is_active' => 'boolean',
        ]);
        $data['slug'] = $data['slug'] ?? Str::slug($data['title']);
        $data['is_featured'] = $request->boolean('is_featured', false);
        $data['is_active'] = $request->boolean('is_active', true);
        $course->update($data);
        return redirect()->route('admin.courses.index')->with('success', 'Course updated.');
    }

    public function destroy(Course $course): RedirectResponse
    {
        $course->delete();
        return redirect()->route('admin.courses.index')->with('success', 'Course deleted.');
    }
}
