<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Course;
use App\Models\CourseEnrollment;
use App\Models\QuizAttempt;
use App\Models\User;
use Illuminate\Support\Facades\DB;

class AdminDashboardController extends Controller
{
    public function __invoke()
    {
        // Overall Stats
        $stats = [
            'total_users' => User::count(),
            'total_students' => User::where('role', 'student')->count(),
            'total_instructors' => User::where('role', 'instructor')->count(),
            'total_courses' => Course::count(),
            'active_courses' => Course::where('is_active', true)->count(),
            'total_enrollments' => CourseEnrollment::count(),
            'completed_enrollments' => CourseEnrollment::where('progress_percent', '>=', 100)->count(),
            'total_revenue' => CourseEnrollment::join('courses', 'course_enrollments.course_id', '=', 'courses.id')
                ->where('courses.type', 'paid')
                ->sum('courses.price'),
            'quiz_attempts' => QuizAttempt::where('status', 'submitted')->count(),
        ];

        // Enrollment Trends (Last 30 days)
        $enrollmentTrends = CourseEnrollment::where('enrolled_at', '>=', now()->subDays(30))
            ->selectRaw('DATE(enrolled_at) as date, COUNT(*) as count')
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Top Performing Courses
        $topCourses = Course::withCount('enrollments')
            ->with('category')
            ->orderBy('enrollments_count', 'desc')
            ->take(5)
            ->get();

        // Recent Activity
        $recentUsers = User::where('role', 'student')->latest()->take(5)->get();
        $recentCourses = Course::with('category')->latest()->take(5)->get();
        $recentEnrollments = CourseEnrollment::with(['user', 'course'])
            ->latest('enrolled_at')
            ->take(5)
            ->get();

        // Course Performance
        $coursePerformance = Course::withCount('enrollments')
            ->withAvg('enrollments', 'progress_percent')
            ->orderBy('enrollments_count', 'desc')
            ->take(10)
            ->get();

        // Monthly Revenue
        $monthlyRevenue = CourseEnrollment::join('courses', 'course_enrollments.course_id', '=', 'courses.id')
            ->where('courses.type', 'paid')
            ->where('course_enrollments.enrolled_at', '>=', now()->subMonths(6))
            ->selectRaw('DATE_FORMAT(course_enrollments.enrolled_at, "%Y-%m") as month, SUM(courses.price) as revenue')
            ->groupBy('month')
            ->orderBy('month')
            ->get();

        return view('admin.dashboard', compact(
            'stats',
            'enrollmentTrends',
            'topCourses',
            'recentUsers',
            'recentCourses',
            'recentEnrollments',
            'coursePerformance',
            'monthlyRevenue'
        ));
    }
}
