<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\View\View;

class AdminInstructorController extends Controller
{
    public function index(Request $request): View
    {
        $query = User::where('role', 'instructor')->withCount('courses');

        if ($request->search) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                  ->orWhere('email', 'like', '%' . $request->search . '%');
            });
        }

        $instructors = $query->latest()->paginate(20)->withQueryString();
        return view('admin.instructors.index', compact('instructors'));
    }

    public function create(): View
    {
        return view('admin.instructors.create');
    }

    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:8',
        ]);

        User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'role' => 'instructor',
        ]);

        return redirect()->route('admin.instructors.index')
            ->with('success', 'Instructor created successfully');
    }

    public function show(User $instructor): View
    {
        if ($instructor->role !== 'instructor') {
            abort(404);
        }
        $instructor->load('courses.category');
        return view('admin.instructors.show', compact('instructor'));
    }

    public function edit(User $instructor): View
    {
        if ($instructor->role !== 'instructor') {
            abort(404);
        }
        return view('admin.instructors.edit', compact('instructor'));
    }

    public function update(Request $request, User $instructor): RedirectResponse
    {
        if ($instructor->role !== 'instructor') {
            abort(404);
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $instructor->id,
            'password' => 'nullable|string|min:8',
        ]);

        $instructor->update([
            'name' => $validated['name'],
            'email' => $validated['email'],
        ]);

        if ($request->filled('password')) {
            $instructor->update(['password' => Hash::make($validated['password'])]);
        }

        return redirect()->route('admin.instructors.index')
            ->with('success', 'Instructor updated successfully');
    }

    public function destroy(User $instructor): RedirectResponse
    {
        if ($instructor->role !== 'instructor') {
            abort(404);
        }
        $instructor->delete();
        return redirect()->route('admin.instructors.index')
            ->with('success', 'Instructor deleted successfully');
    }
}
