<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Course;
use App\Models\CourseLesson;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;

class AdminLessonController extends Controller
{
    public function create(Course $course): View
    {
        return view('admin.lessons.create', compact('course'));
    }

    public function store(Request $request, Course $course): RedirectResponse
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'nullable|string',
            'video_url' => 'nullable|url|max:500',
            'file_url' => 'nullable|url|max:500',
            'duration' => 'required|integer|min:1',
            'sort_order' => 'required|integer|min:0',
            'is_preview' => 'boolean',
            'is_offline_available' => 'boolean',
        ]);

        $validated['course_id'] = $course->id;
        $validated['is_preview'] = $request->has('is_preview');
        $validated['is_offline_available'] = $request->has('is_offline_available');

        CourseLesson::create($validated);

        // Update course totals
        $course->update([
            'total_lessons' => $course->lessons()->count(),
            'total_duration' => $course->lessons()->sum('duration'),
        ]);

        return redirect()->route('admin.courses.show', $course)
            ->with('success', 'Lesson created successfully');
    }

    public function edit(Course $course, CourseLesson $lesson): View
    {
        if ($lesson->course_id !== $course->id) {
            abort(404);
        }
        return view('admin.lessons.edit', compact('course', 'lesson'));
    }

    public function update(Request $request, Course $course, CourseLesson $lesson): RedirectResponse
    {
        if ($lesson->course_id !== $course->id) {
            abort(404);
        }

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'nullable|string',
            'video_url' => 'nullable|url|max:500',
            'file_url' => 'nullable|url|max:500',
            'duration' => 'required|integer|min:1',
            'sort_order' => 'required|integer|min:0',
            'is_preview' => 'boolean',
            'is_offline_available' => 'boolean',
        ]);

        $validated['is_preview'] = $request->has('is_preview');
        $validated['is_offline_available'] = $request->has('is_offline_available');

        $lesson->update($validated);

        // Update course totals
        $course->update([
            'total_lessons' => $course->lessons()->count(),
            'total_duration' => $course->lessons()->sum('duration'),
        ]);

        return redirect()->route('admin.courses.show', $course)
            ->with('success', 'Lesson updated successfully');
    }

    public function destroy(Course $course, CourseLesson $lesson): RedirectResponse
    {
        if ($lesson->course_id !== $course->id) {
            abort(404);
        }

        $lesson->delete();

        // Update course totals
        $course->update([
            'total_lessons' => $course->lessons()->count(),
            'total_duration' => $course->lessons()->sum('duration'),
        ]);

        return redirect()->route('admin.courses.show', $course)
            ->with('success', 'Lesson deleted successfully');
    }
}
