<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\Quiz;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\View\View;

class AdminQuizController extends Controller
{
    public function index(): View
    {
        $quizzes = Quiz::with('category')->latest()->paginate(15);
        return view('admin.quizzes.index', compact('quizzes'));
    }

    public function create(): View
    {
        $categories = Category::where('is_active', true)->orderBy('sort_order')->get();
        return view('admin.quizzes.create', compact('categories'));
    }

    public function store(Request $request): RedirectResponse
    {
        $data = $request->validate([
            'category_id' => 'nullable|exists:categories,id',
            'title' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'subject' => 'nullable|string|max:100',
            'type' => 'required|in:practice,mock,subject_wise,full_length',
            'exam_type' => 'nullable|string|max:100',
            'total_questions' => 'required|integer|min:1',
            'duration' => 'required|integer|min:1',
            'passing_percent' => 'integer|min:0|max:100',
            'price' => 'numeric|min:0',
            'is_free' => 'boolean',
            'is_active' => 'boolean',
        ]);
        $data['slug'] = $data['slug'] ?? Str::slug($data['title']);
        $data['is_free'] = $request->boolean('is_free', true);
        $data['is_active'] = $request->boolean('is_active', true);
        Quiz::create($data);
        return redirect()->route('admin.quizzes.index')->with('success', 'Quiz created.');
    }

    public function show(Quiz $quiz): View
    {
        $quiz->load(['category', 'questions.options']);
        return view('admin.quizzes.show', compact('quiz'));
    }

    public function edit(Quiz $quiz): View
    {
        $categories = Category::where('is_active', true)->orderBy('sort_order')->get();
        return view('admin.quizzes.edit', compact('quiz', 'categories'));
    }

    public function update(Request $request, Quiz $quiz): RedirectResponse
    {
        $data = $request->validate([
            'category_id' => 'nullable|exists:categories,id',
            'title' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'subject' => 'nullable|string|max:100',
            'type' => 'required|in:practice,mock,subject_wise,full_length',
            'exam_type' => 'nullable|string|max:100',
            'total_questions' => 'required|integer|min:1',
            'duration' => 'required|integer|min:1',
            'passing_percent' => 'integer|min:0|max:100',
            'price' => 'numeric|min:0',
            'is_free' => 'boolean',
            'is_active' => 'boolean',
        ]);
        $data['slug'] = $data['slug'] ?? Str::slug($data['title']);
        $data['is_free'] = $request->boolean('is_free', true);
        $data['is_active'] = $request->boolean('is_active', true);
        $quiz->update($data);
        return redirect()->route('admin.quizzes.index')->with('success', 'Quiz updated.');
    }

    public function destroy(Quiz $quiz): RedirectResponse
    {
        $quiz->delete();
        return redirect()->route('admin.quizzes.index')->with('success', 'Quiz deleted.');
    }
}
