<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Course;
use App\Models\CourseEnrollment;
use App\Models\QuizAttempt;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\View\View;

class AdminReportController extends Controller
{
    public function index(Request $request): View
    {
        $dateRange = $request->get('range', '30'); // days
        $startDate = now()->subDays($dateRange);
        $endDate = now();

        // Overall Stats
        $stats = [
            'total_users' => User::count(),
            'total_students' => User::where('role', 'student')->count(),
            'total_instructors' => User::where('role', 'instructor')->count(),
            'total_courses' => Course::count(),
            'active_courses' => Course::where('is_active', true)->count(),
            'total_enrollments' => CourseEnrollment::count(),
            'revenue' => CourseEnrollment::join('courses', 'course_enrollments.course_id', '=', 'courses.id')
                ->where('courses.type', 'paid')
                ->sum('courses.price'),
        ];

        // Enrollment Trends
        $enrollmentTrends = CourseEnrollment::whereBetween('enrolled_at', [$startDate, $endDate])
            ->selectRaw('DATE(enrolled_at) as date, COUNT(*) as count')
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Top Courses
        $topCourses = Course::withCount('enrollments')
            ->orderBy('enrollments_count', 'desc')
            ->take(10)
            ->get();

        // Course Performance
        $coursePerformance = Course::withCount(['enrollments', 'lessons'])
            ->withAvg('enrollments', 'progress_percent')
            ->orderBy('enrollments_count', 'desc')
            ->get();

        // Student Progress
        $studentProgress = CourseEnrollment::with('user', 'course')
            ->orderBy('progress_percent', 'desc')
            ->take(20)
            ->get();

        // Recent Activity
        $recentEnrollments = CourseEnrollment::with(['user', 'course'])
            ->latest('enrolled_at')
            ->take(10)
            ->get();

        return view('admin.reports.index', compact(
            'stats',
            'enrollmentTrends',
            'topCourses',
            'coursePerformance',
            'studentProgress',
            'recentEnrollments',
            'dateRange'
        ));
    }
}
