<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\JsonResponse;
use Illuminate\View\View;

class ApiDocController extends Controller
{
    public function index(): View
    {
        return view('api.docs');
    }

    public function json(): JsonResponse
    {
        $baseUrl = url('/api');

        return response()->json([
            'openapi' => '3.0.0',
            'info' => [
                'title' => config('app.name') . ' API',
                'version' => '1.0.0',
                'description' => 'Complete API documentation for LMS Portal with request and response examples',
                'contact' => [
                    'email' => 'support@lms.test'
                ],
            ],
            'servers' => [
                ['url' => $baseUrl, 'description' => 'Production server'],
            ],
            'paths' => $this->getPaths(),
            'components' => [
                'securitySchemes' => [
                    'bearerAuth' => [
                        'type' => 'http',
                        'scheme' => 'bearer',
                        'bearerFormat' => 'JWT',
                        'description' => 'Enter JWT token obtained from login endpoint',
                    ],
                ],
                'schemas' => $this->getSchemas(),
                'responses' => $this->getResponses(),
            ],
        ]);
    }

    private function getPaths(): array
    {
        return [
            '/auth/register' => [
                'post' => [
                    'tags' => ['Authentication'],
                    'summary' => 'Register a new user',
                    'description' => 'Create a new user account and receive JWT token',
                    'requestBody' => [
                        'required' => true,
                        'content' => [
                            'application/json' => [
                                'schema' => ['$ref' => '#/components/schemas/RegisterRequest'],
                                'example' => [
                                    'name' => 'John Doe',
                                    'email' => 'john@example.com',
                                    'phone' => '+1234567890',
                                    'password' => 'password123',
                                    'password_confirmation' => 'password123',
                                    'role' => 'student',
                                ],
                            ],
                        ],
                    ],
                    'responses' => [
                        '201' => [
                            'description' => 'Registration successful',
                            'content' => [
                                'application/json' => [
                                    'schema' => ['$ref' => '#/components/schemas/AuthResponse'],
                                    'example' => [
                                        'message' => 'Registration successful',
                                        'user' => [
                                            'id' => 1,
                                            'name' => 'John Doe',
                                            'email' => 'john@example.com',
                                            'role' => 'student',
                                        ],
                                        'token' => 'eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9...',
                                        'token_type' => 'bearer',
                                        'expires_in' => 3600,
                                    ],
                                ],
                            ],
                        ],
                        '422' => ['$ref' => '#/components/responses/ValidationError'],
                    ],
                ],
            ],
            '/auth/login' => [
                'post' => [
                    'tags' => ['Authentication'],
                    'summary' => 'Login user',
                    'description' => 'Authenticate user and receive JWT token',
                    'requestBody' => [
                        'required' => true,
                        'content' => [
                            'application/json' => [
                                'schema' => ['$ref' => '#/components/schemas/LoginRequest'],
                                'example' => [
                                    'email' => 'john@example.com',
                                    'password' => 'password123',
                                    'fcm_token' => 'optional-fcm-token',
                                ],
                            ],
                        ],
                    ],
                    'responses' => [
                        '200' => [
                            'description' => 'Login successful',
                            'content' => [
                                'application/json' => [
                                    'schema' => ['$ref' => '#/components/schemas/AuthResponse'],
                                    'example' => [
                                        'message' => 'Login successful',
                                        'user' => [
                                            'id' => 1,
                                            'name' => 'John Doe',
                                            'email' => 'john@example.com',
                                            'role' => 'student',
                                        ],
                                        'token' => 'eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9...',
                                        'token_type' => 'bearer',
                                        'expires_in' => 3600,
                                    ],
                                ],
                            ],
                        ],
                        '401' => ['$ref' => '#/components/responses/Unauthorized'],
                    ],
                ],
            ],
            '/auth/logout' => [
                'post' => [
                    'tags' => ['Authentication'],
                    'summary' => 'Logout user',
                    'description' => 'Invalidate current JWT token',
                    'security' => [['bearerAuth' => []]],
                    'responses' => [
                        '200' => [
                            'description' => 'Logout successful',
                            'content' => [
                                'application/json' => [
                                    'example' => ['message' => 'Logged out successfully'],
                                ],
                            ],
                        ],
                        '401' => ['$ref' => '#/components/responses/Unauthorized'],
                    ],
                ],
            ],
            '/auth/refresh' => [
                'post' => [
                    'tags' => ['Authentication'],
                    'summary' => 'Refresh JWT token',
                    'description' => 'Get a new JWT token using current token',
                    'security' => [['bearerAuth' => []]],
                    'responses' => [
                        '200' => [
                            'description' => 'Token refreshed',
                            'content' => [
                                'application/json' => [
                                    'example' => [
                                        'token' => 'eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9...',
                                        'token_type' => 'bearer',
                                        'expires_in' => 3600,
                                    ],
                                ],
                            ],
                        ],
                        '401' => ['$ref' => '#/components/responses/Unauthorized'],
                    ],
                ],
            ],
            '/auth/me' => [
                'get' => [
                    'tags' => ['Authentication'],
                    'summary' => 'Get current user',
                    'description' => 'Get authenticated user information',
                    'security' => [['bearerAuth' => []]],
                    'responses' => [
                        '200' => [
                            'description' => 'User information',
                            'content' => [
                                'application/json' => [
                                    'schema' => ['$ref' => '#/components/schemas/User'],
                                    'example' => [
                                        'id' => 1,
                                        'name' => 'John Doe',
                                        'email' => 'john@example.com',
                                        'role' => 'student',
                                    ],
                                ],
                            ],
                        ],
                        '401' => ['$ref' => '#/components/responses/Unauthorized'],
                    ],
                ],
            ],
            '/categories' => [
                'get' => [
                    'tags' => ['Categories'],
                    'summary' => 'List all categories',
                    'description' => 'Get list of all active course categories',
                    'responses' => [
                        '200' => [
                            'description' => 'List of categories',
                            'content' => [
                                'application/json' => [
                                    'example' => [
                                        'data' => [
                                            [
                                                'id' => 1,
                                                'name' => 'Data Science',
                                                'slug' => 'data-science',
                                                'description' => 'Data Science courses',
                                                'sort_order' => 1,
                                            ],
                                            [
                                                'id' => 2,
                                                'name' => 'Web Development',
                                                'slug' => 'web-development',
                                                'description' => 'Web Development courses',
                                                'sort_order' => 2,
                                            ],
                                        ],
                                    ],
                                ],
                            ],
                        ],
                    ],
                ],
            ],
            '/categories/{category}' => [
                'get' => [
                    'tags' => ['Categories'],
                    'summary' => 'Get category details',
                    'description' => 'Get detailed information about a specific category',
                    'parameters' => [
                        [
                            'name' => 'category',
                            'in' => 'path',
                            'required' => true,
                            'schema' => ['type' => 'string'],
                            'description' => 'Category slug or ID',
                        ],
                    ],
                    'responses' => [
                        '200' => [
                            'description' => 'Category details',
                            'content' => [
                                'application/json' => [
                                    'schema' => ['$ref' => '#/components/schemas/Category'],
                                    'example' => [
                                        'id' => 1,
                                        'name' => 'Data Science',
                                        'slug' => 'data-science',
                                        'description' => 'Data Science courses',
                                        'sort_order' => 1,
                                        'courses_count' => 15,
                                    ],
                                ],
                            ],
                        ],
                        '404' => ['$ref' => '#/components/responses/NotFound'],
                    ],
                ],
            ],
            '/courses' => [
                'get' => [
                    'tags' => ['Courses'],
                    'summary' => 'List all courses',
                    'description' => 'Get paginated list of active courses with optional filters',
                    'parameters' => [
                        [
                            'name' => 'category_id',
                            'in' => 'query',
                            'schema' => ['type' => 'integer'],
                            'description' => 'Filter by category ID',
                        ],
                        [
                            'name' => 'type',
                            'in' => 'query',
                            'schema' => ['type' => 'string', 'enum' => ['free', 'paid']],
                            'description' => 'Filter by course type',
                        ],
                        [
                            'name' => 'exam_type',
                            'in' => 'query',
                            'schema' => ['type' => 'string'],
                            'description' => 'Filter by exam type',
                        ],
                        [
                            'name' => 'search',
                            'in' => 'query',
                            'schema' => ['type' => 'string'],
                            'description' => 'Search in title and description',
                        ],
                        [
                            'name' => 'page',
                            'in' => 'query',
                            'schema' => ['type' => 'integer'],
                            'description' => 'Page number',
                        ],
                    ],
                    'responses' => [
                        '200' => [
                            'description' => 'List of courses',
                            'content' => [
                                'application/json' => [
                                    'example' => [
                                        'data' => [
                                            [
                                                'id' => 1,
                                                'title' => 'Complete Generative AI Course',
                                                'slug' => 'complete-generative-ai-course',
                                                'description' => 'Learn AI from scratch',
                                                'price' => 4999,
                                                'type' => 'paid',
                                                'rating' => 4.7,
                                                'total_lessons' => 29,
                                                'total_duration' => 1140,
                                                'category' => ['id' => 1, 'name' => 'Data Science'],
                                            ],
                                        ],
                                        'current_page' => 1,
                                        'total' => 10,
                                    ],
                                ],
                            ],
                        ],
                    ],
                ],
            ],
            '/courses/{course}' => [
                'get' => [
                    'tags' => ['Courses'],
                    'summary' => 'Get course details',
                    'description' => 'Get detailed information about a specific course',
                    'parameters' => [
                        [
                            'name' => 'course',
                            'in' => 'path',
                            'required' => true,
                            'schema' => ['type' => 'string'],
                            'description' => 'Course slug or ID',
                        ],
                    ],
                    'responses' => [
                        '200' => [
                            'description' => 'Course details',
                            'content' => [
                                'application/json' => [
                                    'schema' => ['$ref' => '#/components/schemas/Course'],
                                ],
                            ],
                        ],
                        '404' => ['$ref' => '#/components/responses/NotFound'],
                    ],
                ],
            ],
            '/courses/{course}/enroll' => [
                'post' => [
                    'tags' => ['Courses'],
                    'summary' => 'Enroll in a course',
                    'description' => 'Enroll authenticated user in a course (free courses enroll immediately)',
                    'security' => [['bearerAuth' => []]],
                    'parameters' => [
                        [
                            'name' => 'course',
                            'in' => 'path',
                            'required' => true,
                            'schema' => ['type' => 'string'],
                        ],
                    ],
                    'responses' => [
                        '200' => [
                            'description' => 'Enrollment successful',
                            'content' => [
                                'application/json' => [
                                    'example' => ['message' => 'Enrolled successfully'],
                                ],
                            ],
                        ],
                        '400' => [
                            'description' => 'Already enrolled or payment required',
                            'content' => [
                                'application/json' => [
                                    'example' => [
                                        'message' => 'Already enrolled',
                                        // OR
                                        'message' => 'Payment required',
                                        'redirect' => 'payment',
                                        'course_id' => 1,
                                    ],
                                ],
                            ],
                        ],
                        '401' => ['$ref' => '#/components/responses/Unauthorized'],
                    ],
                ],
            ],
            '/courses/{course}/lessons' => [
                'get' => [
                    'tags' => ['Courses'],
                    'summary' => 'Get course lessons',
                    'description' => 'Get all lessons for an enrolled course',
                    'security' => [['bearerAuth' => []]],
                    'parameters' => [
                        [
                            'name' => 'course',
                            'in' => 'path',
                            'required' => true,
                            'schema' => ['type' => 'string'],
                        ],
                    ],
                    'responses' => [
                        '200' => [
                            'description' => 'List of lessons',
                            'content' => [
                                'application/json' => [
                                    'example' => [
                                        'lessons' => [
                                            [
                                                'id' => 1,
                                                'title' => 'Introduction to AI',
                                                'duration' => 25,
                                                'is_preview' => true,
                                                'sort_order' => 1,
                                            ],
                                        ],
                                        'enrollment' => [
                                            'progress_percent' => 0,
                                        ],
                                    ],
                                ],
                            ],
                        ],
                        '403' => [
                            'description' => 'Not enrolled',
                            'content' => [
                                'application/json' => [
                                    'example' => ['error' => 'Not enrolled'],
                                ],
                            ],
                        ],
                        '401' => ['$ref' => '#/components/responses/Unauthorized'],
                    ],
                ],
            ],
            '/lesson-progress' => [
                'post' => [
                    'tags' => ['Courses'],
                    'summary' => 'Update lesson progress',
                    'description' => 'Mark a lesson as completed or update progress',
                    'security' => [['bearerAuth' => []]],
                    'requestBody' => [
                        'required' => true,
                        'content' => [
                            'application/json' => [
                                'schema' => [
                                    'type' => 'object',
                                    'required' => ['course_lesson_id', 'completed'],
                                    'properties' => [
                                        'course_lesson_id' => ['type' => 'integer'],
                                        'completed' => ['type' => 'boolean'],
                                    ],
                                ],
                                'example' => [
                                    'course_lesson_id' => 1,
                                    'completed' => true,
                                ],
                            ],
                        ],
                    ],
                    'responses' => [
                        '200' => [
                            'description' => 'Progress updated',
                            'content' => [
                                'application/json' => [
                                    'example' => ['message' => 'Progress updated'],
                                ],
                            ],
                        ],
                        '401' => ['$ref' => '#/components/responses/Unauthorized'],
                    ],
                ],
            ],
            '/my-courses' => [
                'get' => [
                    'tags' => ['Courses'],
                    'summary' => 'Get enrolled courses',
                    'description' => 'Get all courses enrolled by authenticated user',
                    'security' => [['bearerAuth' => []]],
                    'responses' => [
                        '200' => [
                            'description' => 'List of enrolled courses',
                            'content' => [
                                'application/json' => [
                                    'example' => [
                                        'data' => [
                                            [
                                                'id' => 1,
                                                'course_id' => 1,
                                                'progress_percent' => 45.5,
                                                'enrolled_at' => '2024-01-15T10:00:00.000000Z',
                                                'course' => [
                                                    'id' => 1,
                                                    'title' => 'Complete Generative AI Course',
                                                    'category' => ['name' => 'Data Science'],
                                                ],
                                            ],
                                        ],
                                    ],
                                ],
                            ],
                        ],
                        '401' => ['$ref' => '#/components/responses/Unauthorized'],
                    ],
                ],
            ],
            '/dashboard' => [
                'get' => [
                    'tags' => ['Dashboard'],
                    'summary' => 'Get dashboard data',
                    'description' => 'Get user dashboard statistics and recent activity',
                    'security' => [['bearerAuth' => []]],
                    'responses' => [
                        '200' => [
                            'description' => 'Dashboard data',
                            'content' => [
                                'application/json' => [
                                    'example' => [
                                        'total_courses' => 5,
                                        'completed_courses' => 2,
                                        'in_progress' => 3,
                                        'recent_enrollments' => [],
                                    ],
                                ],
                            ],
                        ],
                        '401' => ['$ref' => '#/components/responses/Unauthorized'],
                    ],
                ],
            ],
            '/quizzes' => [
                'get' => [
                    'tags' => ['Quizzes'],
                    'summary' => 'List all quizzes',
                    'parameters' => [
                        [
                            'name' => 'type',
                            'in' => 'query',
                            'schema' => ['type' => 'string'],
                        ],
                        [
                            'name' => 'subject',
                            'in' => 'query',
                            'schema' => ['type' => 'string'],
                        ],
                        [
                            'name' => 'exam_type',
                            'in' => 'query',
                            'schema' => ['type' => 'string'],
                        ],
                    ],
                    'responses' => [
                        '200' => [
                            'description' => 'List of quizzes',
                            'content' => [
                                'application/json' => [
                                    'example' => [
                                        'data' => [
                                            [
                                                'id' => 1,
                                                'title' => 'Python Basics Quiz',
                                                'description' => 'Test your Python knowledge',
                                                'total_questions' => 20,
                                                'duration' => 30,
                                            ],
                                        ],
                                    ],
                                ],
                            ],
                        ],
                    ],
                ],
            ],
            '/quizzes/{quiz}' => [
                'get' => [
                    'tags' => ['Quizzes'],
                    'summary' => 'Get quiz details',
                    'parameters' => [
                        [
                            'name' => 'quiz',
                            'in' => 'path',
                            'required' => true,
                            'schema' => ['type' => 'integer'],
                        ],
                    ],
                    'responses' => [
                        '200' => [
                            'description' => 'Quiz details',
                            'content' => [
                                'application/json' => [
                                    'schema' => ['$ref' => '#/components/schemas/Quiz'],
                                ],
                            ],
                        ],
                        '404' => ['$ref' => '#/components/responses/NotFound'],
                    ],
                ],
            ],
            '/quizzes/{quiz}/start' => [
                'post' => [
                    'tags' => ['Quizzes'],
                    'summary' => 'Start a quiz',
                    'security' => [['bearerAuth' => []]],
                    'parameters' => [
                        [
                            'name' => 'quiz',
                            'in' => 'path',
                            'required' => true,
                            'schema' => ['type' => 'integer'],
                        ],
                    ],
                    'responses' => [
                        '200' => [
                            'description' => 'Quiz started',
                            'content' => [
                                'application/json' => [
                                    'example' => [
                                        'attempt_id' => 1,
                                        'questions' => [],
                                        'started_at' => '2024-01-15T10:00:00Z',
                                    ],
                                ],
                            ],
                        ],
                        '401' => ['$ref' => '#/components/responses/Unauthorized'],
                    ],
                ],
            ],
            '/quiz-attempts/{attempt}/submit' => [
                'post' => [
                    'tags' => ['Quizzes'],
                    'summary' => 'Submit quiz attempt',
                    'security' => [['bearerAuth' => []]],
                    'parameters' => [
                        [
                            'name' => 'attempt',
                            'in' => 'path',
                            'required' => true,
                            'schema' => ['type' => 'integer'],
                        ],
                    ],
                    'responses' => [
                        '200' => [
                            'description' => 'Quiz submitted',
                            'content' => [
                                'application/json' => [
                                    'example' => [
                                        'score' => 85,
                                        'total_questions' => 20,
                                        'correct_answers' => 17,
                                    ],
                                ],
                            ],
                        ],
                        '401' => ['$ref' => '#/components/responses/Unauthorized'],
                    ],
                ],
            ],
            '/payments/create-order' => [
                'post' => [
                    'tags' => ['Payments'],
                    'summary' => 'Create payment order',
                    'description' => 'Create Razorpay order for course or subscription payment',
                    'security' => [['bearerAuth' => []]],
                    'requestBody' => [
                        'required' => true,
                        'content' => [
                            'application/json' => [
                                'schema' => [
                                    'type' => 'object',
                                    'properties' => [
                                        'course_id' => ['type' => 'integer'],
                                        'subscription_id' => ['type' => 'integer'],
                                        'plan' => ['type' => 'string', 'enum' => ['monthly', 'yearly']],
                                        'coupon_code' => ['type' => 'string'],
                                    ],
                                ],
                                'example' => [
                                    'course_id' => 1,
                                    'coupon_code' => 'DISCOUNT10',
                                ],
                            ],
                        ],
                    ],
                    'responses' => [
                        '200' => [
                            'description' => 'Order created',
                            'content' => [
                                'application/json' => [
                                    'example' => [
                                        'order_id' => 'order_123456',
                                        'amount' => 4999,
                                        'currency' => 'INR',
                                    ],
                                ],
                            ],
                        ],
                        '400' => [
                            'description' => 'Invalid request',
                            'content' => [
                                'application/json' => [
                                    'example' => ['error' => 'Course is free'],
                                ],
                            ],
                        ],
                        '401' => ['$ref' => '#/components/responses/Unauthorized'],
                    ],
                ],
            ],
            '/payments/verify' => [
                'post' => [
                    'tags' => ['Payments'],
                    'summary' => 'Verify payment',
                    'security' => [['bearerAuth' => []]],
                    'requestBody' => [
                        'required' => true,
                        'content' => [
                            'application/json' => [
                                'example' => [
                                    'razorpay_order_id' => 'order_123456',
                                    'razorpay_payment_id' => 'pay_123456',
                                    'razorpay_signature' => 'signature',
                                ],
                            ],
                        ],
                    ],
                    'responses' => [
                        '200' => [
                            'description' => 'Payment verified',
                            'content' => [
                                'application/json' => [
                                    'example' => ['message' => 'Payment successful'],
                                ],
                            ],
                        ],
                        '400' => [
                            'description' => 'Payment verification failed',
                            'content' => [
                                'application/json' => [
                                    'example' => ['error' => 'Invalid signature'],
                                ],
                            ],
                        ],
                        '401' => ['$ref' => '#/components/responses/Unauthorized'],
                    ],
                ],
            ],
            '/profile' => [
                'get' => [
                    'tags' => ['Profile'],
                    'summary' => 'Get user profile',
                    'security' => [['bearerAuth' => []]],
                    'responses' => [
                        '200' => [
                            'description' => 'User profile',
                            'content' => [
                                'application/json' => [
                                    'schema' => ['$ref' => '#/components/schemas/User'],
                                ],
                            ],
                        ],
                        '401' => ['$ref' => '#/components/responses/Unauthorized'],
                    ],
                ],
                'put' => [
                    'tags' => ['Profile'],
                    'summary' => 'Update user profile',
                    'security' => [['bearerAuth' => []]],
                    'requestBody' => [
                        'required' => true,
                        'content' => [
                            'application/json' => [
                                'example' => [
                                    'name' => 'John Doe',
                                    'phone' => '+1234567890',
                                ],
                            ],
                        ],
                    ],
                    'responses' => [
                        '200' => [
                            'description' => 'Profile updated',
                            'content' => [
                                'application/json' => [
                                    'schema' => ['$ref' => '#/components/schemas/User'],
                                ],
                            ],
                        ],
                        '401' => ['$ref' => '#/components/responses/Unauthorized'],
                    ],
                ],
            ],
        ];
    }

    private function getSchemas(): array
    {
        return [
            'RegisterRequest' => [
                'type' => 'object',
                'required' => ['name', 'email', 'password', 'password_confirmation'],
                'properties' => [
                    'name' => ['type' => 'string', 'example' => 'John Doe'],
                    'email' => ['type' => 'string', 'format' => 'email', 'example' => 'john@example.com'],
                    'phone' => ['type' => 'string', 'nullable' => true, 'example' => '+1234567890'],
                    'password' => ['type' => 'string', 'minLength' => 8, 'example' => 'password123'],
                    'password_confirmation' => ['type' => 'string', 'example' => 'password123'],
                    'role' => ['type' => 'string', 'enum' => ['student', 'instructor'], 'default' => 'student'],
                ],
            ],
            'LoginRequest' => [
                'type' => 'object',
                'required' => ['email', 'password'],
                'properties' => [
                    'email' => ['type' => 'string', 'format' => 'email', 'example' => 'john@example.com'],
                    'password' => ['type' => 'string', 'example' => 'password123'],
                    'fcm_token' => ['type' => 'string', 'nullable' => true],
                ],
            ],
            'AuthResponse' => [
                'type' => 'object',
                'properties' => [
                    'message' => ['type' => 'string'],
                    'user' => ['$ref' => '#/components/schemas/User'],
                    'token' => ['type' => 'string'],
                    'token_type' => ['type' => 'string', 'example' => 'bearer'],
                    'expires_in' => ['type' => 'integer', 'example' => 3600],
                ],
            ],
            'User' => [
                'type' => 'object',
                'properties' => [
                    'id' => ['type' => 'integer'],
                    'name' => ['type' => 'string'],
                    'email' => ['type' => 'string'],
                    'phone' => ['type' => 'string', 'nullable' => true],
                    'role' => ['type' => 'string', 'enum' => ['student', 'instructor', 'admin']],
                ],
            ],
            'Course' => [
                'type' => 'object',
                'properties' => [
                    'id' => ['type' => 'integer'],
                    'title' => ['type' => 'string'],
                    'slug' => ['type' => 'string'],
                    'description' => ['type' => 'string'],
                    'price' => ['type' => 'number'],
                    'type' => ['type' => 'string', 'enum' => ['free', 'paid']],
                    'rating' => ['type' => 'number'],
                    'total_lessons' => ['type' => 'integer'],
                    'total_duration' => ['type' => 'integer'],
                    'category' => [
                        'type' => 'object',
                        'properties' => [
                            'id' => ['type' => 'integer'],
                            'name' => ['type' => 'string'],
                        ],
                    ],
                ],
            ],
            'Category' => [
                'type' => 'object',
                'properties' => [
                    'id' => ['type' => 'integer'],
                    'name' => ['type' => 'string'],
                    'slug' => ['type' => 'string'],
                    'description' => ['type' => 'string', 'nullable' => true],
                    'sort_order' => ['type' => 'integer'],
                    'courses_count' => ['type' => 'integer'],
                ],
            ],
            'Quiz' => [
                'type' => 'object',
                'properties' => [
                    'id' => ['type' => 'integer'],
                    'title' => ['type' => 'string'],
                    'description' => ['type' => 'string'],
                    'total_questions' => ['type' => 'integer'],
                    'duration' => ['type' => 'integer'],
                ],
            ],
        ];
    }

    private function getResponses(): array
    {
        return [
            'ValidationError' => [
                'description' => 'Validation error',
                'content' => [
                    'application/json' => [
                        'schema' => [
                            'type' => 'object',
                            'properties' => [
                                'errors' => [
                                    'type' => 'object',
                                    'additionalProperties' => [
                                        'type' => 'array',
                                        'items' => ['type' => 'string'],
                                    ],
                                ],
                            ],
                        ],
                        'example' => [
                            'errors' => [
                                'email' => ['The email field is required.'],
                                'password' => ['The password must be at least 8 characters.'],
                            ],
                        ],
                    ],
                ],
            ],
            'Unauthorized' => [
                'description' => 'Unauthorized',
                'content' => [
                    'application/json' => [
                        'example' => ['error' => 'Unauthenticated'],
                    ],
                ],
            ],
            'NotFound' => [
                'description' => 'Resource not found',
                'content' => [
                    'application/json' => [
                        'example' => ['error' => 'Not found'],
                    ],
                ],
            ],
        ];
    }
}
