<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Course;
use App\Models\CourseEnrollment;
use App\Models\CourseLesson;
use App\Models\LessonProgress;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class CourseController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $query = Course::with('category:id,name,slug')
            ->where('is_active', true);

        if ($request->category_id) {
            $query->where('category_id', $request->category_id);
        }
        if ($request->type) {
            $query->where('type', $request->type);
        }
        if ($request->exam_type) {
            $query->where('exam_type', $request->exam_type);
        }
        if ($request->search) {
            $query->where(function ($q) use ($request) {
                $q->where('title', 'like', "%{$request->search}%")
                    ->orWhere('description', 'like', "%{$request->search}%");
            });
        }

        $courses = $query->latest()->paginate(15);

        return response()->json($courses);
    }

    public function show(Course $course): JsonResponse
    {
        if (!$course->is_active) {
            return response()->json(['error' => 'Course not found'], 404);
        }

        $course->load(['category:id,name,slug', 'instructor:id,name', 'lessons' => fn ($q) => $q->select('id', 'course_id', 'title', 'duration', 'is_preview', 'sort_order')]);
        return response()->json($course);
    }

    public function myCourses(Request $request): JsonResponse
    {
        $enrollments = CourseEnrollment::where('user_id', $request->user()->id)
            ->with('course.category:id,name')
            ->latest('enrolled_at')
            ->paginate(15);

        return response()->json($enrollments);
    }

    public function enroll(Request $request, Course $course): JsonResponse
    {
        $user = $request->user();

        if (CourseEnrollment::where('user_id', $user->id)->where('course_id', $course->id)->exists()) {
            return response()->json(['message' => 'Already enrolled'], 400);
        }

        if ($course->isFree()) {
            CourseEnrollment::create([
                'user_id' => $user->id,
                'course_id' => $course->id,
                'enrolled_at' => now(),
            ]);
            return response()->json(['message' => 'Enrolled successfully']);
        }

        return response()->json(['message' => 'Payment required', 'redirect' => 'payment', 'course_id' => $course->id], 402);
    }

    public function lessons(Request $request, Course $course): JsonResponse
    {
        $enrollment = CourseEnrollment::where('user_id', $request->user()->id)
            ->where('course_id', $course->id)
            ->first();

        if (!$enrollment) {
            return response()->json(['error' => 'Not enrolled'], 403);
        }

        $lessons = $course->lessons;
        $progress = LessonProgress::where('user_id', $request->user()->id)
            ->whereIn('course_lesson_id', $lessons->pluck('id'))
            ->get()
            ->keyBy('course_lesson_id');

        $lessons = $lessons->map(function ($lesson) use ($progress) {
            $lesson->progress = $progress->get($lesson->id);
            return $lesson;
        });

        return response()->json(['lessons' => $lessons, 'enrollment' => $enrollment]);
    }

    public function updateLessonProgress(Request $request): JsonResponse
    {
        $request->validate([
            'lesson_id' => 'required|exists:course_lessons,id',
            'is_completed' => 'boolean',
            'watch_duration' => 'integer|min:0',
        ]);

        $lesson = CourseLesson::findOrFail($request->lesson_id);
        $user = $request->user();

        $enrollment = CourseEnrollment::where('user_id', $user->id)
            ->where('course_id', $lesson->course_id)
            ->firstOrFail();

        $progress = LessonProgress::updateOrCreate(
            ['user_id' => $user->id, 'course_lesson_id' => $lesson->id],
            [
                'is_completed' => $request->boolean('is_completed'),
                'watch_duration' => $request->integer('watch_duration', 0),
                'last_watched_at' => now(),
            ]
        );

        $completedCount = LessonProgress::where('user_id', $user->id)
            ->where('is_completed', true)
            ->whereIn('course_lesson_id', $lesson->course->lessons()->pluck('id'))
            ->count();

        $totalLessons = $lesson->course->lessons()->count();
        $progressPercent = $totalLessons > 0 ? round(($completedCount / $totalLessons) * 100, 2) : 0;
        $enrollment->update(['progress_percent' => $progressPercent]);

        return response()->json(['progress' => $progress, 'enrollment' => $enrollment->fresh()]);
    }
}
