<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Course;
use App\Models\CourseEnrollment;
use App\Models\Coupon;
use App\Models\Payment;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Razorpay\Api\Api;

class PaymentController extends Controller
{
    public function createOrder(Request $request): JsonResponse
    {
        $request->validate([
            'course_id' => 'required_without:subscription_id|exists:courses,id',
            'subscription_id' => 'required_without:course_id|exists:subscriptions,id',
            'plan' => 'required_if:subscription_id,*|in:monthly,yearly',
            'coupon_code' => 'nullable|string',
        ]);

        $user = $request->user();
        $amount = 0;
        $paymentableType = null;
        $paymentableId = null;

        if ($request->course_id) {
            $course = Course::findOrFail($request->course_id);
            if ($course->isFree()) {
                return response()->json(['error' => 'Course is free'], 400);
            }
            if (CourseEnrollment::where('user_id', $user->id)->where('course_id', $course->id)->exists()) {
                return response()->json(['error' => 'Already enrolled'], 400);
            }
            $amount = $course->price;
            $paymentableType = Course::class;
            $paymentableId = $course->id;
        } else {
            $subscription = \App\Models\Subscription::findOrFail($request->subscription_id);
            $amount = (float) ($request->plan === 'yearly' ? $subscription->price_yearly : $subscription->price_monthly);
            $paymentableType = \App\Models\Subscription::class;
            $paymentableId = $subscription->id;
        }

        $discountAmount = 0;
        if ($request->coupon_code) {
            $coupon = Coupon::where('code', $request->coupon_code)->first();
            if ($coupon && $coupon->isValid($amount)) {
                $discountAmount = $coupon->calculateDiscount($amount);
            }
        }
        $finalAmount = max(0, $amount - $discountAmount) * 100; // Razorpay uses paise

        $payment = Payment::create([
            'user_id' => $user->id,
            'paymentable_type' => $paymentableType,
            'paymentable_id' => $paymentableId,
            'amount' => $amount - $discountAmount,
            'payment_method' => 'razorpay',
            'status' => 'pending',
            'coupon_code' => $request->coupon_code,
            'discount_amount' => $discountAmount,
            'metadata' => $request->only(['course_id', 'subscription_id', 'plan']),
        ]);

        $key = config('services.razorpay.key');
        $secret = config('services.razorpay.secret');

        if (!$key || !$secret) {
            return response()->json([
                'payment_id' => $payment->id,
                'order_id' => 'test_order_' . $payment->id,
                'amount' => $finalAmount,
                'message' => 'Configure RAZORPAY_KEY and RAZORPAY_SECRET for live payments',
            ]);
        }

        try {
            $api = new Api($key, $secret);
            $order = $api->order->create([
                'amount' => $finalAmount,
                'currency' => 'INR',
                'receipt' => 'lms_' . $payment->id,
            ]);
            $payment->update(['razorpay_order_id' => $order->id]);
            return response()->json([
                'payment_id' => $payment->id,
                'order_id' => $order->id,
                'amount' => $finalAmount,
                'key' => $key,
            ]);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Payment creation failed: ' . $e->getMessage()], 500);
        }
    }

    public function verify(Request $request): JsonResponse
    {
        $request->validate([
            'payment_id' => 'required|exists:payments,id',
            'razorpay_payment_id' => 'required|string',
            'razorpay_signature' => 'required|string',
        ]);

        $payment = Payment::findOrFail($request->payment_id);
        if ($payment->user_id !== $request->user()->id) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }
        if ($payment->status === 'completed') {
            return response()->json(['message' => 'Already verified', 'payment' => $payment]);
        }

        $key = config('services.razorpay.key');
        $secret = config('services.razorpay.secret');

        if ($key && $secret) {
            try {
                $api = new Api($key, $secret);
                $api->utility->verifyPaymentSignature([
                    'razorpay_order_id' => $payment->razorpay_order_id,
                    'razorpay_payment_id' => $request->razorpay_payment_id,
                    'razorpay_signature' => $request->razorpay_signature,
                ]);
            } catch (\Exception $e) {
                return response()->json(['error' => 'Payment verification failed'], 400);
            }
        }

        $payment->update([
            'razorpay_payment_id' => $request->razorpay_payment_id,
            'razorpay_signature' => $request->razorpay_signature,
            'transaction_id' => $request->razorpay_payment_id,
            'status' => 'completed',
        ]);

        if ($payment->paymentable_type === Course::class) {
            CourseEnrollment::create([
                'user_id' => $payment->user_id,
                'course_id' => $payment->paymentable_id,
                'payment_id' => $payment->id,
                'enrolled_at' => now(),
            ]);
        }

        if ($request->coupon_code) {
            Coupon::where('code', $payment->coupon_code)->increment('used_count');
        }

        return response()->json(['message' => 'Payment verified', 'payment' => $payment->fresh()]);
    }
}
