<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Quiz;
use App\Models\QuizAttempt;
use App\Models\QuizAttemptAnswer;
use App\Models\QuizOption;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class QuizController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $query = Quiz::with('category:id,name')
            ->where('is_active', true);

        if ($request->type) {
            $query->where('type', $request->type);
        }
        if ($request->subject) {
            $query->where('subject', $request->subject);
        }
        if ($request->exam_type) {
            $query->where('exam_type', $request->exam_type);
        }

        $quizzes = $query->latest()->paginate(15);
        return response()->json($quizzes);
    }

    public function show(Quiz $quiz): JsonResponse
    {
        if (!$quiz->is_active) {
            return response()->json(['error' => 'Quiz not found'], 404);
        }
        $quiz->load('category:id,name');
        return response()->json($quiz->makeHidden(['questions']));
    }

    public function start(Request $request, Quiz $quiz): JsonResponse
    {
        $user = $request->user();

        $attempt = QuizAttempt::create([
            'user_id' => $user->id,
            'quiz_id' => $quiz->id,
            'total_questions' => $quiz->total_questions,
            'started_at' => now(),
            'status' => 'in_progress',
        ]);

        $questions = $quiz->questions()
            ->with(['options' => fn ($q) => $q->select('id', 'quiz_question_id', 'option_text', 'option_key')])
            ->get()
            ->map(fn ($q) => [
                'id' => $q->id,
                'question' => $q->question,
                'marks' => $q->marks,
                'options' => $q->options->map(fn ($o) => [
                    'id' => $o->id,
                    'option_key' => $o->option_key,
                    'option_text' => $o->option_text,
                ]),
            ]);

        return response()->json([
            'attempt' => $attempt,
            'quiz' => ['id' => $quiz->id, 'title' => $quiz->title, 'duration' => $quiz->duration],
            'questions' => $questions,
        ]);
    }

    public function saveAnswer(Request $request, QuizAttempt $attempt): JsonResponse
    {
        if ($attempt->user_id !== $request->user()->id || $attempt->status !== 'in_progress') {
            return response()->json(['error' => 'Invalid attempt'], 403);
        }

        $request->validate([
            'question_id' => 'required|exists:quiz_questions,id',
            'option_id' => 'nullable|exists:quiz_options,id',
            'option_key' => 'nullable|string|max:2',
            'time_taken' => 'integer|min:0',
        ]);

        $option = $request->option_id
            ? QuizOption::find($request->option_id)
            : QuizOption::where('quiz_question_id', $request->question_id)
                ->where('option_key', $request->option_key)
                ->first();

        $question = $attempt->quiz->questions()->findOrFail($request->question_id);
        $isCorrect = $option && $option->is_correct;
        $marks = $isCorrect ? $question->marks : ($question->negative_marks ? -$question->negative_marks : 0);

        QuizAttemptAnswer::updateOrCreate(
            ['quiz_attempt_id' => $attempt->id, 'quiz_question_id' => $request->question_id],
            [
                'quiz_option_id' => $option?->id,
                'selected_option_key' => $option?->option_key ?? $request->option_key,
                'is_correct' => $isCorrect,
                'marks_obtained' => $marks,
                'time_taken' => $request->integer('time_taken', 0),
            ]
        );

        return response()->json(['message' => 'Answer saved']);
    }

    public function submit(Request $request, QuizAttempt $attempt): JsonResponse
    {
        if ($attempt->user_id !== $request->user()->id || $attempt->status !== 'in_progress') {
            return response()->json(['error' => 'Invalid attempt'], 403);
        }

        $answers = $attempt->answers;
        $correct = $answers->where('is_correct', true)->count();
        $incorrect = $answers->where('is_correct', false)->count();
        $unanswered = $attempt->total_questions - $answers->count();
        $score = $answers->sum('marks_obtained');
        $totalMarks = $attempt->quiz->questions()->sum('marks');
        $percent = $totalMarks > 0 ? round(($score / $totalMarks) * 100, 2) : 0;

        $attempt->update([
            'attempted' => $answers->count(),
            'correct' => $correct,
            'incorrect' => $incorrect,
            'unanswered' => $unanswered,
            'score' => max(0, $score),
            'percent' => $percent,
            'time_taken' => now()->diffInSeconds($attempt->started_at),
            'submitted_at' => now(),
            'status' => 'submitted',
        ]);

        return response()->json([
            'attempt' => $attempt->fresh(),
            'result' => [
                'correct' => $correct,
                'incorrect' => $incorrect,
                'unanswered' => $unanswered,
                'score' => max(0, $score),
                'percent' => $percent,
                'passed' => $percent >= $attempt->quiz->passing_percent,
            ],
        ]);
    }

    public function result(Request $request, QuizAttempt $attempt): JsonResponse
    {
        if ($attempt->user_id !== $request->user()->id) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $attempt->load(['quiz', 'answers.question.options']);
        $review = $attempt->answers->map(fn ($a) => [
            'question' => $a->question->question,
            'selected' => $a->selected_option_key,
            'correct' => $a->question->options->where('is_correct', true)->first()?->option_key,
            'is_correct' => $a->is_correct,
            'explanation' => $a->question->explanation,
        ]);

        return response()->json([
            'attempt' => $attempt,
            'review' => $review,
        ]);
    }
}
