<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Schedule;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class ScheduleController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $query = Schedule::where('user_id', $request->user()->id);

        if ($request->date) {
            $query->where('schedule_date', $request->date);
        }
        if ($request->type) {
            $query->where('schedule_type', $request->type);
        }
        if ($request->is_completed !== null) {
            $query->where('is_completed', $request->boolean('is_completed'));
        }

        $schedules = $query->with(['course:id,title', 'quiz:id,title'])
            ->orderBy('schedule_date')
            ->orderBy('start_time')
            ->paginate(20);

        $progress = null;
        if ($request->show_progress) {
            $total = Schedule::where('user_id', $request->user()->id)->count();
            $completed = Schedule::where('user_id', $request->user()->id)->where('is_completed', true)->count();
            $progress = ['total' => $total, 'completed' => $completed, 'percent' => $total ? round(($completed / $total) * 100, 1) : 0];
        }

        return response()->json(['schedules' => $schedules, 'progress' => $progress]);
    }

    public function store(Request $request): JsonResponse
    {
        $data = $request->validate([
            'title' => 'required|string|max:255',
            'course_id' => 'nullable|exists:courses,id',
            'course_lesson_id' => 'nullable|exists:course_lessons,id',
            'quiz_id' => 'nullable|exists:quizzes,id',
            'subject' => 'nullable|string|max:100',
            'schedule_date' => 'required|date',
            'start_time' => 'required|date_format:H:i',
            'end_time' => 'nullable|date_format:H:i',
            'schedule_type' => 'required|in:day,week,month',
            'notes' => 'nullable|string',
            'reminder_enabled' => 'boolean',
        ]);

        $data['user_id'] = $request->user()->id;
        $data['reminder_enabled'] = $request->boolean('reminder_enabled', true);

        $schedule = Schedule::create($data);
        return response()->json($schedule->load(['course:id,title', 'quiz:id,title']), 201);
    }

    public function show(Request $request, Schedule $schedule): JsonResponse
    {
        if ($schedule->user_id !== $request->user()->id) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }
        $schedule->load(['course', 'lesson', 'quiz']);
        return response()->json($schedule);
    }

    public function update(Request $request, Schedule $schedule): JsonResponse
    {
        if ($schedule->user_id !== $request->user()->id) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $data = $request->validate([
            'title' => 'string|max:255',
            'course_id' => 'nullable|exists:courses,id',
            'course_lesson_id' => 'nullable|exists:course_lessons,id',
            'quiz_id' => 'nullable|exists:quizzes,id',
            'subject' => 'nullable|string|max:100',
            'schedule_date' => 'date',
            'start_time' => 'date_format:H:i',
            'end_time' => 'nullable|date_format:H:i',
            'schedule_type' => 'in:day,week,month',
            'notes' => 'nullable|string',
            'reminder_enabled' => 'boolean',
            'is_completed' => 'boolean',
        ]);

        if (isset($data['is_completed']) && $data['is_completed']) {
            $data['completed_at'] = now();
        }

        $schedule->update($data);
        return response()->json($schedule->fresh());
    }

    public function destroy(Request $request, Schedule $schedule): JsonResponse
    {
        if ($schedule->user_id !== $request->user()->id) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }
        $schedule->delete();
        return response()->json(null, 204);
    }
}
