<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use App\Models\Course;
use App\Models\CourseEnrollment;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;

class WebCourseController extends Controller
{
    public function index(Request $request): View
    {
        $query = Course::with('category')->where('is_active', true);
        if ($request->category) {
            $query->whereHas('category', fn ($q) => $q->where('slug', $request->category));
        }
        if ($request->type === 'free') {
            $query->where('type', 'free');
        }
        $courses = $query->latest()->paginate(12);
        $categories = \App\Models\Category::where('is_active', true)->orderBy('sort_order')->get();
        return view('courses.index', compact('courses', 'categories'));
    }

    public function show(Course $course): View
    {
        if (!$course->is_active) {
            abort(404);
        }
        $course->load(['category', 'lessons', 'instructor']);
        $enrolled = auth()->check() && CourseEnrollment::where('user_id', auth()->id())
            ->where('course_id', $course->id)->exists();

        // Related courses (same category, exclude current)
        $relatedCourses = Course::where('is_active', true)
            ->where('id', '!=', $course->id)
            ->where('category_id', $course->category_id)
            ->with('category')
            ->take(6)
            ->get();

        // If not enough related courses, get any active courses
        if ($relatedCourses->count() < 3) {
            $relatedCourses = Course::where('is_active', true)
                ->where('id', '!=', $course->id)
                ->with('category')
                ->take(6)
                ->get();
        }

        // Testimonials (can be filtered by course/program later)
        $testimonials = \App\Models\Testimonial::where('is_active', true)
            ->orderBy('sort_order')
            ->take(6)
            ->get();

        return view('courses.show', compact('course', 'enrolled', 'relatedCourses', 'testimonials'));
    }

    public function enroll(Course $course): RedirectResponse
    {
        $user = auth()->user();
        if (CourseEnrollment::where('user_id', $user->id)->where('course_id', $course->id)->exists()) {
            return redirect()->route('courses.show', $course)->with('message', 'Already enrolled');
        }
        if ($course->isFree()) {
            CourseEnrollment::create([
                'user_id' => $user->id,
                'course_id' => $course->id,
                'enrolled_at' => now(),
            ]);
            return redirect()->route('courses.my-courses')->with('message', 'Enrolled successfully');
        }
        return redirect()->route('courses.show', $course)->with('message', 'Payment required for paid courses');
    }

    public function myCourses(): View
    {
        $enrollments = CourseEnrollment::where('user_id', auth()->id())
            ->with('course.category')
            ->whereHas('course', function($query) {
                $query->whereNotNull('slug')->where('slug', '!=', '');
            })
            ->latest('enrolled_at')
            ->paginate(12);
        return view('courses.my-courses', compact('enrollments'));
    }

    public function learn(Course $course): View|RedirectResponse
    {
        $enrollment = CourseEnrollment::where('user_id', auth()->id())
            ->where('course_id', $course->id)
            ->first();

        // If not enrolled, try to auto-enroll if course is free
        if (!$enrollment && $course->isFree()) {
            CourseEnrollment::create([
                'user_id' => auth()->id(),
                'course_id' => $course->id,
                'enrolled_at' => now(),
            ]);
            $enrollment = CourseEnrollment::where('user_id', auth()->id())
                ->where('course_id', $course->id)
                ->first();
        }

        if (!$enrollment) {
            return redirect()->route('courses.show', $course)
                ->with('error', 'You must enroll in this course first. Click "Enroll Now" to get started!');
        }

        $course->load(['category', 'lessons', 'instructor']);
        $firstLesson = $course->lessons->sortBy('sort_order')->first();

        if ($firstLesson) {
            return redirect()->route('courses.lesson', ['course' => $course, 'lesson' => $firstLesson]);
        }

        return view('courses.learn', compact('course', 'enrollment'));
    }

    public function lesson(Course $course, \App\Models\CourseLesson $lesson): View|RedirectResponse
    {
        // Check enrollment (still required for full access; preview handled below)
        $enrollment = CourseEnrollment::where('user_id', auth()->id())
            ->where('course_id', $course->id)
            ->first();

        // Allow preview lessons without enrollment, but require enrollment for others
        if (!$enrollment && !$lesson->is_preview) {
            return redirect()->route('courses.show', $course)
                ->with('error', 'You must enroll in this course to access this lesson. Enroll now to continue learning!');
        }

        $course->load(['category', 'lessons', 'instructor']);
        $allLessons = $course->lessons->sortBy('sort_order')->values();
        $currentIndex = $allLessons->search(fn($l) => $l->id === $lesson->id);
        $prevLesson = $currentIndex > 0 ? $allLessons[$currentIndex - 1] : null;
        $nextLesson = $currentIndex < $allLessons->count() - 1 ? $allLessons[$currentIndex + 1] : null;

        // Group lessons into sections/modules intelligently
        // Try to group by natural breaks (every 4-6 lessons) or by lesson title patterns
        $sections = [];
        $lessonsPerSection = max(4, min(6, ceil($allLessons->count() / max(4, floor($allLessons->count() / 6)))));
        $sectionNum = 0;
        $currentSectionLessons = [];

        foreach ($allLessons as $index => $l) {
            // Check if this lesson starts a new section (by title pattern or count)
            $shouldStartNewSection = empty($currentSectionLessons) ||
                count($currentSectionLessons) >= $lessonsPerSection ||
                (str_contains(strtolower($l->title), 'section') ||
                 str_contains(strtolower($l->title), 'module') ||
                 str_contains(strtolower($l->title), 'project') ||
                 str_contains(strtolower($l->title), 'deployment'));

            if ($shouldStartNewSection && !empty($currentSectionLessons)) {
                $sectionNum++;
                $firstLessonTitle = $currentSectionLessons[0]->title;
                // Extract section name from first lesson or use generic
                $sectionTitle = $firstLessonTitle;
                if (preg_match('/^(Section \d+|Module \d+)[:\-]?\s*(.+)$/i', $firstLessonTitle, $matches)) {
                    $sectionTitle = trim($matches[1] . ': ' . $matches[2]);
                } elseif (strlen($firstLessonTitle) > 40) {
                    $sectionTitle = 'Section ' . $sectionNum . ': ' . substr($firstLessonTitle, 0, 30) . '...';
                } else {
                    $sectionTitle = 'Section ' . $sectionNum . ': ' . $firstLessonTitle;
                }

                $sections[] = [
                    'title' => $sectionTitle,
                    'lessons' => $currentSectionLessons,
                ];
                $currentSectionLessons = [];
            }

            $currentSectionLessons[] = $l;
        }

        // Add remaining lessons
        if (!empty($currentSectionLessons)) {
            $sectionNum++;
            $firstLessonTitle = $currentSectionLessons[0]->title;
            $sectionTitle = strlen($firstLessonTitle) > 40
                ? 'Section ' . $sectionNum . ': ' . substr($firstLessonTitle, 0, 30) . '...'
                : 'Section ' . $sectionNum . ': ' . $firstLessonTitle;

            $sections[] = [
                'title' => $sectionTitle,
                'lessons' => $currentSectionLessons,
            ];
        }

        return view('courses.learn', compact('course', 'lesson', 'enrollment', 'sections', 'prevLesson', 'nextLesson', 'allLessons'));
    }
}
