<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;

class Course extends Model
{
    use HasFactory;

    protected $fillable = [
        'category_id',
        'instructor_id',
        'title',
        'slug',
        'description',
        'syllabus',
        'thumbnail',
        'demo_video_url',
        'price',
        'type',
        'rating',
        'total_ratings',
        'total_lessons',
        'total_duration',
        'level',
        'exam_type',
        'is_featured',
        'is_active',
    ];

    protected function casts(): array
    {
        return [
            'price' => 'decimal:2',
            'rating' => 'decimal:2',
            'is_featured' => 'boolean',
            'is_active' => 'boolean',
        ];
    }

    protected static function booted(): void
    {
        static::creating(function (self $course): void {
            if (empty($course->slug) && ! empty($course->title)) {
                $base = Str::slug($course->title);
                $course->slug = $base ?: Str::random(8);
            }
        });

        static::updating(function (self $course): void {
            if (empty($course->slug) && ! empty($course->title)) {
                $base = Str::slug($course->title);
                $course->slug = $base ?: Str::random(8);
            }
        });
    }

    public function category(): BelongsTo
    {
        return $this->belongsTo(Category::class);
    }

    public function instructor(): BelongsTo
    {
        return $this->belongsTo(User::class, 'instructor_id');
    }

    public function lessons(): HasMany
    {
        return $this->hasMany(CourseLesson::class)->orderBy('sort_order');
    }

    public function enrollments(): HasMany
    {
        return $this->hasMany(CourseEnrollment::class);
    }

    public function students(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'course_enrollments')
            ->withPivot('progress_percent', 'enrolled_at')
            ->withTimestamps();
    }

    public function ratings(): HasMany
    {
        return $this->hasMany(CourseRating::class);
    }

    public function isFree(): bool
    {
        return $this->type === 'free' || $this->price <= 0;
    }

    public function getRouteKeyName(): string
    {
        return 'slug';
    }
}
