<?php

namespace Database\Seeders;

use App\Models\Category;
use App\Models\Course;
use App\Models\CourseLesson;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Str;

class CourseSeeder extends Seeder
{
    public function run(): void
    {
        // Get or create instructor
        $instructor = User::firstOrCreate(
            ['email' => 'instructor@lms.test'],
            [
                'name' => 'Dr. Sarah Johnson',
                'password' => bcrypt('password'),
                'role' => 'instructor',
            ]
        );

        // Get categories
        $categories = Category::all();
        if ($categories->isEmpty()) {
            $categories = collect([
                Category::create(['name' => 'Data Science', 'slug' => 'data-science', 'sort_order' => 1]),
                Category::create(['name' => 'Web Development', 'slug' => 'web-development', 'sort_order' => 2]),
                Category::create(['name' => 'Machine Learning', 'slug' => 'machine-learning', 'sort_order' => 3]),
                Category::create(['name' => 'Business', 'slug' => 'business', 'sort_order' => 4]),
            ]);
        }

        // Course 1: Complete Generative AI Course
        $course1 = Course::create([
            'category_id' => $categories->first()->id,
            'instructor_id' => $instructor->id,
            'title' => 'Complete Generative AI Course: RAG, AI Agents & Deployment',
            'slug' => 'complete-generative-ai-course-rag-ai-agents-deployment',
            'description' => 'Learn Generative AI from scratch - Build RAG, AI Agents & Chatbots, master MCP, and deploy real-world projects. This comprehensive course covers everything from foundations to advanced deployment strategies.',
            'syllabus' => "Master the foundations of Generative AI and Large Language Models (LLMs)\nUse real-world AI applications including RAG, AI Agents, and Chatbots\nBuild production-ready AI systems with deployment strategies\nImplement Retrieval-Augmented Generation (RAG) for enhanced AI responses\nCreate intelligent AI agents that can think, plan, and act\nMaster Model Context Protocol (MCP) for better AI interactions\nDeploy AI applications to production environments\nIntegrate AI solutions across multiple domains (Retail, BFSI, Healthcare)\nWork with 32+ modern AI tools and frameworks\nGain hands-on experience with industry-grade projects",
            'price' => 4999,
            'type' => 'paid',
            'rating' => 4.7,
            'total_ratings' => 1250,
            'level' => 'intermediate',
            'exam_type' => 'Professional Certification',
            'is_featured' => true,
            'is_active' => true,
        ]);

        // Lessons for Course 1 - Section 1: Introduction
        $lessons1 = [
            ['title' => 'Introduction to Generative AI', 'duration' => 25, 'content' => 'Overview of Generative AI, its applications, and current landscape.'],
            ['title' => 'Understanding Large Language Models', 'duration' => 30, 'content' => 'Deep dive into LLMs, their architecture, and capabilities.'],
            ['title' => 'Setting Up Your Development Environment', 'duration' => 20, 'content' => 'Install and configure all necessary tools and frameworks.'],
            ['title' => 'First AI Application: Hello World', 'duration' => 15, 'content' => 'Build your first simple AI application.'],
        ];

        // Section 2: Generative AI - Foundation
        $lessons2 = [
            ['title' => 'Core Concepts of Generative AI', 'duration' => 35, 'content' => 'Understanding the fundamental concepts and principles.'],
            ['title' => 'Transformer Architecture Explained', 'duration' => 40, 'content' => 'Deep dive into transformer models and attention mechanisms.'],
            ['title' => 'Prompt Engineering Fundamentals', 'duration' => 30, 'content' => 'Learn how to craft effective prompts for better AI responses.'],
            ['title' => 'Fine-tuning vs. Prompt Engineering', 'duration' => 25, 'content' => 'Compare different approaches to optimizing AI models.'],
            ['title' => 'Hands-on: Building Your First Chatbot', 'duration' => 45, 'content' => 'Practical exercise building a functional chatbot.'],
        ];

        // Section 3: Accessing LLMs in Python
        $lessons3 = [
            ['title' => 'Open LLMs vs Proprietary Models', 'duration' => 30, 'content' => 'Compare open-source and proprietary LLM options.'],
            ['title' => 'API Integration with OpenAI', 'duration' => 35, 'content' => 'Learn to integrate OpenAI APIs in your applications.'],
            ['title' => 'Working with Local Models (Llama, Mistral)', 'duration' => 40, 'content' => 'Set up and use local LLM models for privacy and cost control.'],
            ['title' => 'LangChain Framework Overview', 'duration' => 30, 'content' => 'Introduction to LangChain for building AI applications.'],
            ['title' => 'Building Multi-Model Applications', 'duration' => 35, 'content' => 'Create applications that use multiple AI models.'],
        ];

        // Section 4: RAG Implementation
        $lessons4 = [
            ['title' => 'Introduction to RAG (Retrieval-Augmented Generation)', 'duration' => 30, 'content' => 'Understanding RAG architecture and benefits.'],
            ['title' => 'Vector Databases and Embeddings', 'duration' => 40, 'content' => 'Working with vector databases for semantic search.'],
            ['title' => 'Building a RAG System from Scratch', 'duration' => 50, 'content' => 'Step-by-step implementation of a complete RAG system.'],
            ['title' => 'Advanced RAG Techniques', 'duration' => 45, 'content' => 'Advanced patterns and optimizations for RAG systems.'],
            ['title' => 'Project: Document Q&A System', 'duration' => 60, 'content' => 'Build a production-ready document question-answering system.'],
        ];

        // Section 5: AI Agents
        $lessons5 = [
            ['title' => 'Understanding AI Agents', 'duration' => 35, 'content' => 'Introduction to AI agents and their capabilities.'],
            ['title' => 'Agent Architecture: Think, Plan, Act', 'duration' => 40, 'content' => 'Deep dive into how agents think, plan, and execute actions.'],
            ['title' => 'Building Your First AI Agent', 'duration' => 45, 'content' => 'Hands-on tutorial building a functional AI agent.'],
            ['title' => 'Multi-Agent Systems', 'duration' => 50, 'content' => 'Creating systems with multiple collaborating agents.'],
            ['title' => 'Project: E-commerce AI Assistant', 'duration' => 60, 'content' => 'Build an AI agent for e-commerce customer support.'],
        ];

        // Section 6: Deployment & Production
        $lessons6 = [
            ['title' => 'Deployment Strategies for AI Applications', 'duration' => 40, 'content' => 'Overview of deployment options and best practices.'],
            ['title' => 'Containerization with Docker', 'duration' => 35, 'content' => 'Containerize your AI applications for easy deployment.'],
            ['title' => 'Cloud Deployment (AWS, GCP, Azure)', 'duration' => 45, 'content' => 'Deploy AI applications to major cloud platforms.'],
            ['title' => 'Monitoring and Scaling AI Systems', 'duration' => 40, 'content' => 'Monitor performance and scale your AI applications.'],
            ['title' => 'Final Project: Deploy a Production AI System', 'duration' => 90, 'content' => 'Complete project deploying a full-stack AI application.'],
        ];

        $allLessons1 = array_merge($lessons1, $lessons2, $lessons3, $lessons4, $lessons5, $lessons6);
        $totalDuration1 = array_sum(array_column($allLessons1, 'duration'));

        foreach ($allLessons1 as $index => $lesson) {
            CourseLesson::create([
                'course_id' => $course1->id,
                'title' => $lesson['title'],
                'content' => $lesson['content'],
                'duration' => $lesson['duration'],
                'sort_order' => $index + 1,
                'is_preview' => $index === 0,
                'video_url' => 'https://sample-videos.com/video123/mp4/720/big_buck_bunny_720p_1mb.mp4',
            ]);
        }

        $course1->update([
            'total_lessons' => count($allLessons1),
            'total_duration' => $totalDuration1,
        ]);

        // Course 2: Full Stack Web Development
        $course2 = Course::create([
            'category_id' => $categories->skip(1)->first()->id,
            'instructor_id' => $instructor->id,
            'title' => 'Full Stack Web Development with React & Node.js',
            'slug' => 'full-stack-web-development-react-nodejs',
            'description' => 'Master modern web development with React frontend and Node.js backend. Build real-world applications from scratch.',
            'syllabus' => "Build responsive web applications with React\nMaster Node.js and Express for backend development\nImplement RESTful APIs and GraphQL\nWork with databases (MongoDB, PostgreSQL)\nDeploy applications to production\nImplement authentication and authorization\nUse modern development tools and workflows\nBuild real-world projects",
            'price' => 3999,
            'type' => 'paid',
            'rating' => 4.6,
            'total_ratings' => 890,
            'level' => 'beginner',
            'exam_type' => 'Professional Certification',
            'is_featured' => true,
            'is_active' => true,
        ]);

        $lessons2_course = [
            ['title' => 'Introduction to Web Development', 'duration' => 20],
            ['title' => 'HTML5 & CSS3 Fundamentals', 'duration' => 30],
            ['title' => 'JavaScript Basics', 'duration' => 35],
            ['title' => 'React Fundamentals', 'duration' => 40],
            ['title' => 'React Hooks and State Management', 'duration' => 45],
            ['title' => 'Node.js Introduction', 'duration' => 30],
            ['title' => 'Express.js Framework', 'duration' => 35],
            ['title' => 'Database Integration', 'duration' => 40],
            ['title' => 'Authentication & Security', 'duration' => 45],
            ['title' => 'Deployment Strategies', 'duration' => 35],
            ['title' => 'Project: E-commerce Platform', 'duration' => 120],
        ];

        $totalDuration2 = array_sum(array_column($lessons2_course, 'duration'));
        foreach ($lessons2_course as $index => $lesson) {
            CourseLesson::create([
                'course_id' => $course2->id,
                'title' => $lesson['title'],
                'content' => 'Learn ' . $lesson['title'] . ' with hands-on examples and practical exercises.',
                'duration' => $lesson['duration'],
                'sort_order' => $index + 1,
                'is_preview' => $index === 0,
            ]);
        }

        $course2->update([
            'total_lessons' => count($lessons2_course),
            'total_duration' => $totalDuration2,
        ]);

        // Course 3: Machine Learning Fundamentals (Free)
        $course3 = Course::create([
            'category_id' => $categories->skip(2)->first()->id,
            'instructor_id' => $instructor->id,
            'title' => 'Machine Learning Fundamentals for Beginners',
            'slug' => 'machine-learning-fundamentals-beginners',
            'description' => 'Learn the fundamentals of machine learning with Python. Perfect for beginners starting their ML journey.',
            'syllabus' => "Introduction to Machine Learning\nPython for Data Science\nData Preprocessing and Cleaning\nSupervised Learning Algorithms\nUnsupervised Learning\nModel Evaluation and Validation\nIntroduction to Deep Learning\nHands-on Projects",
            'price' => 0,
            'type' => 'free',
            'rating' => 4.5,
            'total_ratings' => 2100,
            'level' => 'beginner',
            'exam_type' => 'Certificate of Completion',
            'is_featured' => false,
            'is_active' => true,
        ]);

        $lessons3_course = [
            ['title' => 'What is Machine Learning?', 'duration' => 25],
            ['title' => 'Python Setup and Libraries', 'duration' => 20],
            ['title' => 'Data Exploration with Pandas', 'duration' => 30],
            ['title' => 'Linear Regression', 'duration' => 35],
            ['title' => 'Logistic Regression', 'duration' => 30],
            ['title' => 'Decision Trees and Random Forests', 'duration' => 40],
            ['title' => 'Clustering Algorithms', 'duration' => 35],
            ['title' => 'Model Evaluation Metrics', 'duration' => 30],
            ['title' => 'Project: House Price Prediction', 'duration' => 60],
        ];

        $totalDuration3 = array_sum(array_column($lessons3_course, 'duration'));
        foreach ($lessons3_course as $index => $lesson) {
            CourseLesson::create([
                'course_id' => $course3->id,
                'title' => $lesson['title'],
                'content' => 'Comprehensive guide to ' . $lesson['title'] . ' with practical examples.',
                'duration' => $lesson['duration'],
                'sort_order' => $index + 1,
                'is_preview' => $index === 0,
            ]);
        }

        $course3->update([
            'total_lessons' => count($lessons3_course),
            'total_duration' => $totalDuration3,
        ]);

        // Course 4: Business Analytics
        $course4 = Course::create([
            'category_id' => $categories->last()->id,
            'instructor_id' => $instructor->id,
            'title' => 'Business Analytics and Data-Driven Decision Making',
            'slug' => 'business-analytics-data-driven-decision-making',
            'description' => 'Learn to analyze business data and make data-driven decisions. Master Excel, SQL, and visualization tools.',
            'syllabus' => "Introduction to Business Analytics\nExcel Advanced Functions\nSQL for Data Analysis\nData Visualization with Tableau\nStatistical Analysis\nPredictive Analytics\nBusiness Intelligence Tools\nCase Studies and Real-world Applications",
            'price' => 2999,
            'type' => 'paid',
            'rating' => 4.4,
            'total_ratings' => 650,
            'level' => 'intermediate',
            'exam_type' => 'Professional Certification',
            'is_featured' => false,
            'is_active' => true,
        ]);

        $lessons4_course = [
            ['title' => 'Introduction to Business Analytics', 'duration' => 25],
            ['title' => 'Excel Advanced Functions', 'duration' => 40],
            ['title' => 'SQL Fundamentals', 'duration' => 35],
            ['title' => 'Data Visualization Principles', 'duration' => 30],
            ['title' => 'Tableau Basics', 'duration' => 40],
            ['title' => 'Statistical Analysis', 'duration' => 45],
            ['title' => 'Predictive Modeling', 'duration' => 50],
            ['title' => 'Case Study: Sales Analysis', 'duration' => 60],
        ];

        $totalDuration4 = array_sum(array_column($lessons4_course, 'duration'));
        foreach ($lessons4_course as $index => $lesson) {
            CourseLesson::create([
                'course_id' => $course4->id,
                'title' => $lesson['title'],
                'content' => 'Master ' . $lesson['title'] . ' for business intelligence.',
                'duration' => $lesson['duration'],
                'sort_order' => $index + 1,
                'is_preview' => $index === 0,
            ]);
        }

        $course4->update([
            'total_lessons' => count($lessons4_course),
            'total_duration' => $totalDuration4,
        ]);

        $this->command->info('Created 4 comprehensive courses with ' .
            (count($allLessons1) + count($lessons2_course) + count($lessons3_course) + count($lessons4_course)) .
            ' total lessons!');
    }
}
