<?php

namespace Database\Seeders;

use App\Models\Category;
use App\Models\Course;
use App\Models\CourseLesson;
use App\Models\CourseEnrollment;
use App\Models\Quiz;
use App\Models\QuizQuestion;
use App\Models\QuizOption;
use App\Models\Subscription;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class DatabaseSeeder extends Seeder
{
    public function run(): void
    {
        // Admin user
        User::create([
            'name' => 'Admin User',
            'email' => 'admin@lms.test',
            'password' => Hash::make('password'),
            'role' => 'admin',
        ]);

        // Sample student
        $student = User::create([
            'name' => 'Test Student',
            'email' => 'student@lms.test',
            'password' => Hash::make('password'),
            'role' => 'student',
        ]);

        // Categories
        $categories = [
            ['name' => 'SSC CGL', 'slug' => 'ssc-cgl'],
            ['name' => 'Banking', 'slug' => 'banking'],
            ['name' => 'Railway', 'slug' => 'railway'],
            ['name' => 'NEET', 'slug' => 'neet'],
            ['name' => 'JEE', 'slug' => 'jee'],
        ];
        foreach ($categories as $i => $c) {
            Category::create([
                'name' => $c['name'],
                'slug' => $c['slug'],
                'sort_order' => $i + 1,
            ]);
        }

        // Sample courses
        $cat = Category::first();
        $courses = [
            ['title' => 'SSC Quantitative Aptitude', 'price' => 999, 'type' => 'paid'],
            ['title' => 'Banking English', 'price' => 0, 'type' => 'free'],
            ['title' => 'Railway General Awareness', 'price' => 499, 'type' => 'paid'],
        ];
        foreach ($courses as $c) {
            $course = Course::create([
                'category_id' => $cat->id,
                'title' => $c['title'],
                'slug' => Str::slug($c['title']),
                'description' => 'Complete course for ' . $c['title'],
                'price' => $c['price'],
                'type' => $c['type'],
                'exam_type' => 'SSC',
                'total_lessons' => 5,
                'total_duration' => 600,
            ]);
            for ($i = 1; $i <= 5; $i++) {
                CourseLesson::create([
                    'course_id' => $course->id,
                    'title' => "Lesson {$i}: Introduction",
                    'content' => "Lesson {$i} content...",
                    'duration' => 30,
                    'sort_order' => $i,
                    'is_preview' => $i === 1,
                ]);
            }
        }

        // Enroll student in free course
        $freeCourse = Course::where('type', 'free')->first();
        if ($freeCourse) {
            CourseEnrollment::create([
                'user_id' => $student->id,
                'course_id' => $freeCourse->id,
                'progress_percent' => 20,
                'enrolled_at' => now(),
            ]);
        }

        // Quizzes
        $quizzes = [
            ['title' => 'Quantitative Aptitude - Practice Test', 'type' => 'practice', 'subject' => 'Math'],
            ['title' => 'Banking Mock Test 1', 'type' => 'mock', 'subject' => 'General'],
        ];
        foreach ($quizzes as $q) {
            $quiz = Quiz::create([
                'category_id' => $cat->id,
                'title' => $q['title'],
                'slug' => Str::slug($q['title']),
                'subject' => $q['subject'],
                'type' => $q['type'],
                'total_questions' => 5,
                'duration' => 30,
                'passing_percent' => 40,
                'is_free' => true,
            ]);
            for ($i = 1; $i <= 5; $i++) {
                $question = QuizQuestion::create([
                    'quiz_id' => $quiz->id,
                    'question' => "Sample question {$i}?",
                    'marks' => 1,
                    'sort_order' => $i,
                ]);
                foreach (['A', 'B', 'C', 'D'] as $j => $key) {
                    QuizOption::create([
                        'quiz_question_id' => $question->id,
                        'option_text' => "Option {$key}",
                        'option_key' => $key,
                        'is_correct' => $j === 0,
                        'sort_order' => $j + 1,
                    ]);
                }
            }
        }

        // Subscription plans
        Subscription::create([
            'name' => 'Basic',
            'slug' => 'basic',
            'price_monthly' => 299,
            'price_yearly' => 2990,
            'features' => ['Access to free courses', 'Practice quizzes'],
        ]);
        Subscription::create([
            'name' => 'Premium',
            'slug' => 'premium',
            'price_monthly' => 599,
            'price_yearly' => 5990,
            'features' => ['All courses', 'Mock tests', 'Live classes'],
            'premium_tests' => true,
        ]);

        $this->call(TestimonialSeeder::class);
        $this->call(CourseSeeder::class);
    }
}
