<?php

use App\Http\Controllers\Api\ApiDocController;
use App\Http\Controllers\Api\AuthController;
use App\Http\Controllers\Api\CategoryController;
use App\Http\Controllers\Api\CourseController;
use App\Http\Controllers\Api\DashboardController;
use App\Http\Controllers\Api\PaymentController;
use App\Http\Controllers\Api\ProfileController;
use App\Http\Controllers\Api\QuizController;
use App\Http\Controllers\Api\ScheduleController;
use Illuminate\Support\Facades\Route;

// API Documentation
Route::get('/docs', [ApiDocController::class, 'index'])->name('api.docs');
Route::get('/docs.json', [ApiDocController::class, 'json'])->name('api.docs.json');

// API Root - API Information
Route::get('/', function () {
    return response()->json([
        'name' => config('app.name') . ' API',
        'version' => '1.0.0',
        'status' => 'active',
        'endpoints' => [
            'auth' => [
                'register' => 'POST /api/auth/register',
                'login' => 'POST /api/auth/login',
                'logout' => 'POST /api/auth/logout',
                'refresh' => 'POST /api/auth/refresh',
                'me' => 'GET /api/auth/me',
            ],
            'categories' => [
                'list' => 'GET /api/categories',
                'show' => 'GET /api/categories/{category}',
            ],
            'courses' => [
                'list' => 'GET /api/courses',
                'show' => 'GET /api/courses/{course}',
                'enroll' => 'POST /api/courses/{course}/enroll',
                'lessons' => 'GET /api/courses/{course}/lessons',
                'my-courses' => 'GET /api/my-courses',
            ],
            'quizzes' => [
                'list' => 'GET /api/quizzes',
                'show' => 'GET /api/quizzes/{quiz}',
                'start' => 'POST /api/quizzes/{quiz}/start',
                'submit' => 'POST /api/quiz-attempts/{attempt}/submit',
            ],
            'profile' => [
                'show' => 'GET /api/profile',
                'update' => 'PUT /api/profile',
            ],
            'payments' => [
                'create-order' => 'POST /api/payments/create-order',
                'verify' => 'POST /api/payments/verify',
            ],
        ],
        'documentation' => 'API documentation available at /api',
    ]);
});

// Public routes
Route::post('/auth/register', [AuthController::class, 'register']);
Route::post('/auth/login', [AuthController::class, 'login']);

// Public course/quiz listing (no auth)
Route::get('/categories', [CategoryController::class, 'index']);
Route::get('/categories/{category}', [CategoryController::class, 'show']);
Route::get('/courses', [CourseController::class, 'index']);
Route::get('/courses/{course}', [CourseController::class, 'show']);
Route::get('/quizzes', [QuizController::class, 'index']);
Route::get('/quizzes/{quiz}', [QuizController::class, 'show']);

// Protected routes (JWT)
Route::middleware('auth:api')->group(function () {
    Route::post('/auth/logout', [AuthController::class, 'logout']);
    Route::post('/auth/refresh', [AuthController::class, 'refresh']);
    Route::get('/auth/me', [AuthController::class, 'me']);

    Route::get('/dashboard', [DashboardController::class, 'index']);
    Route::get('/my-courses', [CourseController::class, 'myCourses']);
    Route::post('/courses/{course}/enroll', [CourseController::class, 'enroll']);
    Route::get('/courses/{course}/lessons', [CourseController::class, 'lessons']);
    Route::post('/lesson-progress', [CourseController::class, 'updateLessonProgress']);

    Route::post('/quizzes/{quiz}/start', [QuizController::class, 'start']);
    Route::post('/quiz-attempts/{attempt}/submit', [QuizController::class, 'submit']);
    Route::post('/quiz-attempts/{attempt}/answer', [QuizController::class, 'saveAnswer']);
    Route::get('/quiz-attempts/{attempt}/result', [QuizController::class, 'result']);

    Route::apiResource('schedules', ScheduleController::class);

    Route::post('/payments/create-order', [PaymentController::class, 'createOrder']);
    Route::post('/payments/verify', [PaymentController::class, 'verify']);

    Route::get('/profile', [ProfileController::class, 'show']);
    Route::put('/profile', [ProfileController::class, 'update']);
});
