<?php

namespace App\Http\Controllers;

use App\Models\Document;
use App\Models\Approval;
use App\Models\Notification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ApprovalController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Show documents pending approval for the current user.
     */
    public function index()
    {
        $user = Auth::user();
        $documents = collect();

        if ($user->hasRole('Consultant')) {
            $documents = Document::where('current_status', 'Pending Consultant Review')
                ->with(['contractor', 'versions', 'approvals'])
                ->get();
        } elseif ($user->hasRole('ClientA')) {
            $documents = Document::where('current_status', 'Pending Client A Review')
                ->with(['contractor', 'versions', 'approvals'])
                ->get();
        } elseif ($user->hasRole(['ClientB', 'ClientC'])) {
            $documents = Document::where('current_status', 'Pending Client B & C Review')
                ->with(['contractor', 'versions', 'approvals'])
                ->get();
        }

        return view('approvals.index', compact('documents'));
    }

    /**
     * Show the approval form for a specific document.
     */
    public function show($id)
    {
        $document = Document::with(['contractor', 'versions', 'approvals.reviewer'])->findOrFail($id);
        $user = Auth::user();

        // Check if user can review this document based on current status
        $canReview = false;
        if ($user->hasRole('Consultant') && $document->current_status === 'Pending Consultant Review') {
            $canReview = true;
        } elseif ($user->hasRole('ClientA') && $document->current_status === 'Pending Client A Review') {
            $canReview = true;
        } elseif ($user->hasRole(['ClientB', 'ClientC']) && $document->current_status === 'Pending Client B & C Review') {
            $canReview = true;
        }

        if (!$canReview) {
            abort(403, 'You cannot review this document at this stage.');
        }

        return view('approvals.show', compact('document'));
    }

    /**
     * Process the approval/rejection.
     */
    public function process(Request $request, $id)
    {
        $document = Document::findOrFail($id);
        $user = Auth::user();

        $request->validate([
            'action' => 'required|in:approve,reject',
            'comments' => 'nullable|string|max:1000',
        ]);

        // Check if user can review this document
        $canReview = false;
        $userRole = '';

        if ($user->hasRole('Consultant') && $document->current_status === 'Pending Consultant Review') {
            $canReview = true;
            $userRole = 'Consultant';
        } elseif ($user->hasRole('ClientA') && $document->current_status === 'Pending Client A Review') {
            $canReview = true;
            $userRole = 'ClientA';
        } elseif ($user->hasRole(['ClientB', 'ClientC']) && $document->current_status === 'Pending Client B & C Review') {
            $canReview = true;
            $userRole = $user->hasRole('ClientB') ? 'ClientB' : 'ClientC';
        }

        if (!$canReview) {
            abort(403, 'You cannot review this document at this stage.');
        }

        DB::beginTransaction();
        try {
            // Create approval record
            $approval = Approval::create([
                'document_id' => $document->id,
                'reviewed_by' => $user->id,
                'role' => $userRole,
                'status' => $request->action === 'approve' ? 'approved' : 'rejected',
                'comments' => $request->comments,
            ]);

            // Update document status based on the workflow
            $newStatus = $this->determineNewStatus($document, $request->action, $userRole);
            $document->update(['current_status' => $newStatus]);

            // Send notifications
            $this->sendNotifications($document, $request->action, $userRole, $request->comments);

            DB::commit();

            $message = $request->action === 'approve' ? 'Document approved successfully!' : 'Document rejected successfully!';
            return redirect()->route('approvals.index')->with('success', $message);
        } catch (\Exception $e) {
            DB::rollback();
            return back()->withErrors(['error' => 'Failed to process approval. Please try again.']);
        }
    }

    /**
     * Determine the new status based on the workflow logic.
     */
    private function determineNewStatus($document, $action, $userRole)
    {
        if ($action === 'reject') {
            return "Rejected by {$userRole}";
        }

        // Approval logic
        switch ($userRole) {
            case 'Consultant':
                return 'Pending Client A Review';
            case 'ClientA':
                return 'Pending Client B & C Review';
            case 'ClientB':
            case 'ClientC':
                // Check if the other client (B or C) has already approved
                $otherClientRole = $userRole === 'ClientB' ? 'ClientC' : 'ClientB';
                $otherClientApproval = Approval::where('document_id', $document->id)
                    ->where('role', $otherClientRole)
                    ->where('status', 'approved')
                    ->exists();

                if ($otherClientApproval) {
                    return 'Final Approved';
                }

                return 'Pending Client B & C Review'; // Still waiting for the other client
        }
    }

    /**
     * Send notifications based on the approval action.
     */
    private function sendNotifications($document, $action, $userRole, $comments)
    {
        // Always notify the contractor
        $message = '';
        if ($action === 'approve') {
            $message = "Your document '{$document->title}' has been approved by {$userRole}.";
        } else {
            $message = "Your document '{$document->title}' has been rejected by {$userRole}.";
            if ($comments) {
                $message .= " Comments: {$comments}";
            }
        }

        Notification::create([
            'user_id' => $document->contractor_id,
            'document_id' => $document->id,
            'message' => $message,
        ]);

        // If approved, notify the next reviewers
        if ($action === 'approve') {
            if ($userRole === 'Consultant') {
                // Notify Client A
                $clientAs = \App\Models\User::role('ClientA')->get();
                foreach ($clientAs as $clientA) {
                    Notification::create([
                        'user_id' => $clientA->id,
                        'document_id' => $document->id,
                        'message' => "Document '{$document->title}' has been approved by Consultant and requires your review.",
                    ]);
                }
            } elseif ($userRole === 'ClientA') {
                // Notify Client B & C
                $clients = \App\Models\User::role(['ClientB', 'ClientC'])->get();
                foreach ($clients as $client) {
                    Notification::create([
                        'user_id' => $client->id,
                        'document_id' => $document->id,
                        'message' => "Document '{$document->title}' has been approved by Client A and requires your review.",
                    ]);
                }
            }
        }
    }
}
