<?php

namespace App\Http\Controllers;

use App\Models\Document;
use App\Models\Approval;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index()
    {
        $user = Auth::user();

        if ($user->hasRole('SuperAdmin')) {
            return $this->superAdminDashboard();
        } elseif ($user->hasRole('Contractor')) {
            return $this->contractorDashboard();
        } elseif ($user->hasRole('Consultant')) {
            return $this->consultantDashboard();
        } elseif ($user->hasRole('ClientA')) {
            return $this->clientADashboard();
        } elseif ($user->hasRole(['ClientB', 'ClientC'])) {
            return $this->clientBCDashboard();
        }

        return $this->defaultDashboard();
    }

    private function superAdminDashboard()
    {
        // System Overview Statistics
        $totalUsers = User::count();
        $totalDocuments = Document::count();
        $totalApprovals = Approval::count();
        $totalNotifications = \App\Models\Notification::count();

        // User distribution by role
        $usersByRole = User::select('roles.name as role_name', DB::raw('count(*) as count'))
            ->join('model_has_roles', 'users.id', '=', 'model_has_roles.model_id')
            ->join('roles', 'model_has_roles.role_id', '=', 'roles.id')
            ->where('model_has_roles.model_type', 'App\\Models\\User')
            ->groupBy('roles.name')
            ->get();

        // Document status distribution
        $documentsByStatus = Document::select('current_status', DB::raw('count(*) as count'))
            ->groupBy('current_status')
            ->get();

        // Document types distribution
        $documentsByType = Document::select('type', DB::raw('count(*) as count'))
            ->groupBy('type')
            ->get();

        // Workflow funnel data
        $workflowFunnel = [
            'submitted' => Document::count(),
            'consultant_approved' => Document::whereIn('current_status', [
                'Pending Client A Review',
                'Pending Client B & C Review',
                'Final Approved'
            ])->count(),
            'client_a_approved' => Document::whereIn('current_status', [
                'Pending Client B & C Review',
                'Final Approved'
            ])->count(),
            'final_approved' => Document::where('current_status', 'Final Approved')->count(),
        ];

        // Recent system activity
        $recentDocuments = Document::with(['contractor', 'versions'])
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        // Approval time analytics
        $averageApprovalTimes = Approval::select('role',
                DB::raw('AVG(TIMESTAMPDIFF(DAY, (SELECT created_at FROM documents WHERE id = approvals.document_id), approvals.created_at)) as avg_days'))
            ->groupBy('role')
            ->get();

        // Monthly document trends
        $monthlyDocuments = Document::select(
                DB::raw('MONTH(created_at) as month'),
                DB::raw('YEAR(created_at) as year'),
                DB::raw('count(*) as count')
            )
            ->where('created_at', '>=', now()->subMonths(12))
            ->groupBy('year', 'month')
            ->orderBy('year', 'asc')
            ->orderBy('month', 'asc')
            ->get();

        // Users without roles
        $usersWithoutRoles = User::doesntHave('roles')->count();

        return view('dashboard', compact(
            'totalUsers',
            'totalDocuments',
            'totalApprovals',
            'totalNotifications',
            'usersByRole',
            'documentsByStatus',
            'documentsByType',
            'workflowFunnel',
            'recentDocuments',
            'averageApprovalTimes',
            'monthlyDocuments',
            'usersWithoutRoles'
        ));
    }

    private function contractorDashboard()
    {
        $user = Auth::user();

        // Document statistics
        $totalDocuments = $user->documents()->count();
        $pendingConsultant = $user->documents()->where('current_status', 'Pending Consultant Review')->count();
        $rejectedDocuments = $user->documents()->where('current_status', 'like', 'Rejected by%')->count();
        $approvedDocuments = $user->documents()->where('current_status', 'Final Approved')->count();

        // Recent submissions
        $recentSubmissions = $user->documents()
            ->with(['versions', 'approvals.reviewer'])
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();

        // Document categories for pie chart
        $documentCategories = $user->documents()
            ->select('type', DB::raw('count(*) as count'))
            ->groupBy('type')
            ->get();

        // Status distribution
        $statusDistribution = $user->documents()
            ->select('current_status', DB::raw('count(*) as count'))
            ->groupBy('current_status')
            ->get();

        return view('dashboard', compact(
            'totalDocuments',
            'pendingConsultant',
            'rejectedDocuments',
            'approvedDocuments',
            'recentSubmissions',
            'documentCategories',
            'statusDistribution'
        ));
    }

    private function consultantDashboard()
    {
        $user = Auth::user();

        // Pending reviews
        $pendingReviews = Document::where('current_status', 'Pending Consultant Review')->count();

        // Approval statistics
        $totalApprovals = $user->approvals()->count();
        $approved = $user->approvals()->where('status', 'approved')->count();
        $rejected = $user->approvals()->where('status', 'rejected')->count();

        // Documents awaiting action
        $documentsAwaitingAction = Document::where('current_status', 'Pending Consultant Review')
            ->with(['contractor', 'versions'])
            ->orderBy('created_at', 'asc')
            ->limit(10)
            ->get();

        // Recent notifications (approvals/rejections)
        $recentNotifications = $user->notifications()
            ->with('document')
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();

        // Weekly approval trends
        $weeklyApprovals = $user->approvals()
            ->select(
                DB::raw('WEEK(created_at) as week'),
                DB::raw('SUM(CASE WHEN status = "approved" THEN 1 ELSE 0 END) as approved'),
                DB::raw('SUM(CASE WHEN status = "rejected" THEN 1 ELSE 0 END) as rejected')
            )
            ->where('created_at', '>=', now()->subWeeks(8))
            ->groupBy('week')
            ->orderBy('week', 'asc')
            ->get();

        return view('dashboard', compact(
            'pendingReviews',
            'totalApprovals',
            'approved',
            'rejected',
            'documentsAwaitingAction',
            'recentNotifications',
            'weeklyApprovals'
        ));
    }

    private function clientADashboard()
    {
        $user = Auth::user();

        // Queue of documents approved by consultant
        $queueCount = Document::where('current_status', 'Pending Client A Review')->count();

        // Documents in queue
        $documentsInQueue = Document::where('current_status', 'Pending Client A Review')
            ->with(['contractor', 'versions', 'approvals'])
            ->orderBy('created_at', 'asc')
            ->get();

        // Approval statistics
        $totalApprovals = $user->approvals()->count();
        $approved = $user->approvals()->where('status', 'approved')->count();
        $rejected = $user->approvals()->where('status', 'rejected')->count();

        // Weekly approval trends
        $weeklyApprovals = $user->approvals()
            ->select(
                DB::raw('WEEK(created_at) as week'),
                DB::raw('SUM(CASE WHEN status = "approved" THEN 1 ELSE 0 END) as approved'),
                DB::raw('SUM(CASE WHEN status = "rejected" THEN 1 ELSE 0 END) as rejected')
            )
            ->where('created_at', '>=', now()->subWeeks(8))
            ->groupBy('week')
            ->orderBy('week', 'asc')
            ->get();

        return view('dashboard', compact(
            'queueCount',
            'documentsInQueue',
            'totalApprovals',
            'approved',
            'rejected',
            'weeklyApprovals'
        ));
    }

    private function clientBCDashboard()
    {
        $user = Auth::user();

        // Documents pending final review
        $pendingReview = Document::where('current_status', 'Pending Client B & C Review')->count();

        // Documents awaiting final review
        $documentsAwaitingReview = Document::where('current_status', 'Pending Client B & C Review')
            ->with(['contractor', 'versions', 'approvals'])
            ->orderBy('created_at', 'asc')
            ->get();

        // Approval statistics
        $totalApprovals = $user->approvals()->count();
        $approved = $user->approvals()->where('status', 'approved')->count();
        $rejected = $user->approvals()->where('status', 'rejected')->count();

        // Monthly approval trends (stacked bar chart data)
        $monthlyApprovals = $user->approvals()
            ->select(
                DB::raw('MONTH(created_at) as month'),
                DB::raw('YEAR(created_at) as year'),
                DB::raw('SUM(CASE WHEN status = "approved" THEN 1 ELSE 0 END) as approved'),
                DB::raw('SUM(CASE WHEN status = "rejected" THEN 1 ELSE 0 END) as rejected')
            )
            ->where('created_at', '>=', now()->subMonths(6))
            ->groupBy('year', 'month')
            ->orderBy('year', 'asc')
            ->orderBy('month', 'asc')
            ->get();

        return view('dashboard', compact(
            'pendingReview',
            'documentsAwaitingReview',
            'totalApprovals',
            'approved',
            'rejected',
            'monthlyApprovals'
        ));
    }

    private function defaultDashboard()
    {
        $user = Auth::user();

        // Basic statistics for users without specific roles
        $totalUsers = User::count();
        $totalDocuments = Document::count();
        $totalApprovals = Approval::count();

        $recentNotifications = $user->notifications()
            ->with('document')
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();

        return view('dashboard', compact(
            'totalUsers',
            'totalDocuments',
            'totalApprovals',
            'recentNotifications'
        ));
    }

    public function getWorkflowFunnel()
    {
        // Admin feature: Workflow funnel data
        $funnelData = [
            'contractor_submissions' => Document::count(),
            'consultant_approved' => Document::whereIn('current_status', [
                'Pending Client A Review',
                'Pending Client B & C Review',
                'Final Approved'
            ])->count(),
            'client_a_approved' => Document::whereIn('current_status', [
                'Pending Client B & C Review',
                'Final Approved'
            ])->count(),
            'final_approved' => Document::where('current_status', 'Final Approved')->count(),
        ];

        return response()->json($funnelData);
    }

    public function getApprovalTimes()
    {
        // Admin feature: Average approval times per role
        $approvalTimes = Approval::select('role', DB::raw('AVG(DATEDIFF(created_at, (SELECT created_at FROM documents WHERE id = approvals.document_id))) as avg_days'))
            ->groupBy('role')
            ->get();

        return response()->json($approvalTimes);
    }
}
