<?php

namespace App\Http\Controllers;

use App\Models\Document;
use App\Models\DocumentVersion;
use App\Models\Notification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

class DocumentController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $user = Auth::user();

        if ($user->hasRole('Contractor')) {
            $documents = $user->documents()->with(['versions', 'approvals.reviewer'])->get();
        } else {
            $documents = Document::with(['contractor', 'versions', 'approvals.reviewer'])->get();
        }

        return view('documents.index', compact('documents'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        if (!Auth::user()->hasRole('Contractor')) {
            abort(403, 'Only contractors can upload documents.');
        }
        return view('documents.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        if (!Auth::user()->hasRole('Contractor')) {
            abort(403, 'Only contractors can upload documents.');
        }

        $request->validate([
            'title' => 'required|string|max:255',
            'type' => 'required|in:Design,MAS,TQ,ST,Material Inspection',
            'file' => 'required|file|mimes:pdf,doc,docx,jpg,png|max:10240', // 10MB max
        ]);

        DB::beginTransaction();
        try {
            // Store the file
            $file = $request->file('file');
            $filename = time() . '_' . $file->getClientOriginalName();
            $filePath = $file->storeAs('documents', $filename);

            // Create the document
            $document = Document::create([
                'title' => $request->title,
                'type' => $request->type,
                'file_path' => $filePath,
                'current_status' => 'Pending Consultant Review',
                'contractor_id' => Auth::id(),
            ]);

            // Create the first version
            DocumentVersion::create([
                'document_id' => $document->id,
                'file_path' => $filePath,
                'version_number' => 1,
                'uploaded_by' => Auth::id(),
            ]);

            // Notify consultants
            $this->notifyConsultants($document);

            DB::commit();
            return redirect()->route('documents.index')->with('success', 'Document uploaded successfully!');
        } catch (\Exception $e) {
            DB::rollback();
            return back()->withErrors(['error' => 'Failed to upload document. Please try again.']);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $document = Document::with(['contractor', 'versions.uploader', 'approvals.reviewer'])->findOrFail($id);

        // Check permissions
        if (Auth::user()->hasRole('Contractor') && $document->contractor_id !== Auth::id()) {
            abort(403, 'You can only view your own documents.');
        }

        return view('documents.show', compact('document'));
    }

    /**
     * Show document history.
     */
    public function history($id)
    {
        $document = Document::with(['versions.uploader', 'approvals.reviewer'])->findOrFail($id);

        // Check permissions
        if (Auth::user()->hasRole('Contractor') && $document->contractor_id !== Auth::id()) {
            abort(403, 'You can only view your own documents.');
        }

        return view('documents.history', compact('document'));
    }

    /**
     * Upload a new version of the document.
     */
    public function uploadVersion(Request $request, $id)
    {
        $document = Document::findOrFail($id);

        // Only contractors can upload new versions of their own documents
        if (!Auth::user()->hasRole('Contractor') || $document->contractor_id !== Auth::id()) {
            abort(403, 'You can only upload new versions of your own documents.');
        }

        $request->validate([
            'file' => 'required|file|mimes:pdf,doc,docx,jpg,png|max:10240', // 10MB max
        ]);

        DB::beginTransaction();
        try {
            // Store the new file
            $file = $request->file('file');
            $filename = time() . '_' . $file->getClientOriginalName();
            $filePath = $file->storeAs('documents', $filename);

            // Get the next version number
            $latestVersion = $document->versions()->max('version_number');
            $newVersionNumber = $latestVersion + 1;

            // Create new version
            DocumentVersion::create([
                'document_id' => $document->id,
                'file_path' => $filePath,
                'version_number' => $newVersionNumber,
                'uploaded_by' => Auth::id(),
            ]);

            // Update document file path and reset status
            $document->update([
                'file_path' => $filePath,
                'current_status' => 'Pending Consultant Review',
            ]);

            // Notify consultants
            $this->notifyConsultants($document, true);

            DB::commit();
            return redirect()->route('documents.show', $document)->with('success', 'New version uploaded successfully!');
        } catch (\Exception $e) {
            DB::rollback();
            return back()->withErrors(['error' => 'Failed to upload new version. Please try again.']);
        }
    }

    /**
     * Download a document file.
     */
    public function download($id, $versionId = null)
    {
        $document = Document::findOrFail($id);

        // Check permissions
        if (Auth::user()->hasRole('Contractor') && $document->contractor_id !== Auth::id()) {
            abort(403, 'You can only download your own documents.');
        }

        if ($versionId) {
            $version = DocumentVersion::where('document_id', $id)->findOrFail($versionId);
            $filePath = $version->file_path;
        } else {
            $filePath = $document->file_path;
        }

        if (!Storage::exists($filePath)) {
            abort(404, 'File not found.');
        }

        return Storage::download($filePath);
    }

    /**
     * Notify consultants about new document or version.
     */
    private function notifyConsultants(Document $document, $isNewVersion = false)
    {
        $consultants = \App\Models\User::role('Consultant')->get();
        $message = $isNewVersion
            ? "New version of document '{$document->title}' has been uploaded and requires your review."
            : "New document '{$document->title}' has been uploaded and requires your review.";

        foreach ($consultants as $consultant) {
            Notification::create([
                'user_id' => $consultant->id,
                'document_id' => $document->id,
                'message' => $message,
            ]);
        }
    }
}
