<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Spatie\Permission\Models\Role;

class UserController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of users.
     */
    public function index()
    {
        if (!Auth::user()->hasRole(['SuperAdmin', 'Consultant', 'ClientA'])) {
            abort(403, 'You do not have permission to manage users.');
        }

        $users = User::with('roles')->paginate(20);
        return view('users.index', compact('users'));
    }

    /**
     * Show the form for assigning a role to a user.
     */
    public function showAssignRole(User $user)
    {
        if (!Auth::user()->hasRole(['SuperAdmin', 'Consultant', 'ClientA'])) {
            abort(403, 'You do not have permission to assign roles.');
        }

        $roles = Role::all();
        return view('users.assign-role', compact('user', 'roles'));
    }

    /**
     * Assign a role to a user.
     */
    public function assignRole(Request $request, User $user)
    {
        if (!Auth::user()->hasRole(['SuperAdmin', 'Consultant', 'ClientA'])) {
            abort(403, 'You do not have permission to assign roles.');
        }

        $request->validate([
            'role' => 'required|exists:roles,name',
        ]);

        // Remove all existing roles first
        $user->syncRoles([]);

        // Assign the new role
        $user->assignRole($request->role);

        return redirect()->route('users.index')->with('success', "Role '{$request->role}' assigned to {$user->name} successfully!");
    }

    /**
     * Show the form for creating a new user.
     */
    public function create()
    {
        if (!Auth::user()->hasRole('SuperAdmin')) {
            abort(403, 'Only SuperAdmin can create users.');
        }

        $roles = Role::all();
        return view('users.create', compact('roles'));
    }

    /**
     * Store a newly created user.
     */
    public function store(Request $request)
    {
        if (!Auth::user()->hasRole('SuperAdmin')) {
            abort(403, 'Only SuperAdmin can create users.');
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'role' => 'required|exists:roles,name',
        ]);

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => bcrypt($request->password),
        ]);

        $user->assignRole($request->role);

        return redirect()->route('users.index')->with('success', "User '{$user->name}' created successfully with role '{$request->role}'!");
    }

    /**
     * Delete a user.
     */
    public function destroy(User $user)
    {
        if (!Auth::user()->hasRole('SuperAdmin')) {
            abort(403, 'Only SuperAdmin can delete users.');
        }

        if ($user->id === Auth::id()) {
            return redirect()->route('users.index')->with('error', 'You cannot delete yourself!');
        }

        $userName = $user->name;
        $user->delete();

        return redirect()->route('users.index')->with('success', "User '{$userName}' deleted successfully!");
    }
}
